#ifndef ga_h
#define ga_h

#include <stdlib.h>
#include <stdio.h>
#include <algorithm>

using namespace std;

// the length of a string representing and individual
const unsigned STR_LEN=23;

//// there are 26 letters in the English alphabet
//const unsigned NUM_LETTERS=26;

// function generating a random character in the interval 'a'..'z'
inline char gen_rnd_char(){
	return char(rand()%NUM_LETTERS)+'a';
}

// class representing an individual
class Individual{
public:
	// the string representing the individual
	char* str;

	// the measure of fitness of an individual
	unsigned score;

	// the number of the individual in order of generation
	unsigned number;

	// a random individual
	Individual(){
		str=new char[STR_LEN+1];
		str[STR_LEN]=0;
		score=0;
		number=0;
	}

	~Individual(){
		delete str;
	}

	void random(){
		for(unsigned i=0;i<STR_LEN;i++){
			str[i]=gen_rnd_char();
		}
	}

	void copy(Individual* ind){
		// copy the individual
		memcpy(str,ind->str,STR_LEN);
		score=ind->score;
	}

	// an individual obtained as the result 
	// of mutation of another individual
	void mutate(Individual* ind){
		const unsigned MUTATION_RATE=1;
		char ch;

		// copy the individual
		copy(ind);
		// and mutate some of the genes;
		// on average, MUTATION_RATE letters are mutated
		for(unsigned i=0;i<STR_LEN;i++){
			if(rand()%STR_LEN<MUTATION_RATE){
				// generate a letter different from str[i]
				while( (ch=gen_rnd_char())==str[i] );
				str[i]=ch;
			}
		}
	}
};


// a population of size at most POPULATION_SIZE;
// represented as a binary heap for efficient 
// "removal of the unfit"
class Population{
protected:
	static bool compare_individuals(Individual* i1, Individual* i2){
		return i1->score>i2->score;
	}

	static int compare_individuals_num(const void* ppi1, const void* ppi2){
		return int((*(Individual**)ppi1)->number)-int((*(Individual**)ppi2)->number);
	}

	// auxiliary variables for computing next generation
	unsigned counter;
	Individual** next_generation;
	Individual* current_placeholder;

	// a set of Individuals which have the best score in the 
	// current generation and thus have to be printed
	Individual** print_buffer;

	// true if there is improvement in fitness over the previous generation
	bool improvement;

	// checks if there are better individuals in the population,
	// and ammends best_individual if necessary
	void ammend_best_individual(Individual* ind){
		if(compare_individuals(ind,best_individual)){
			improvement=true;
			best_individual->copy(ind);
		}
	}

	typedef void Tscore_hook(Individual* ind);
	typedef void Tprint_hook(Individual* ind);

	// pointer to a function computing the fitness of an individual
	Tscore_hook* score_hook;
	// pointer to a function printing an individual in a nice format
	Tprint_hook* print_hook;

public:
	// the size of the population
	unsigned POPULATION_SIZE;

	unsigned max_score;

	// a set of Individuals
	Individual** individuals;

	// the best individual encountered so far
	Individual* best_individual;

	// the total number of individuals created so far
	unsigned long total;

	// the number of the currenct generation
	unsigned long generation_number;

	Population(unsigned pop_size, Tscore_hook* ascore_hook, Tprint_hook* aprint_hook, unsigned amax_score){
		POPULATION_SIZE=pop_size;
		score_hook=ascore_hook;
		print_hook=aprint_hook;
		max_score=amax_score;

		improvement=false;
		counter=0;
		total=0;
		generation_number=0;
		current_placeholder=0;

		individuals=new Individual*[POPULATION_SIZE];
		next_generation=new Individual*[POPULATION_SIZE];
		print_buffer=new Individual*[POPULATION_SIZE];

		unsigned i;

		// generate POPULATION_SIZE random individuals
		for(i=0;i<POPULATION_SIZE;i++){
			individuals[i]=new Individual();
			individuals[i]->random();
			score_hook(individuals[i]);
			individuals[i]->number=++total;

			// these individuals are just `placeholders'
			next_generation[i]=new Individual();
			next_generation[i]->random();
		}

		// generate a placeholder for the best individual,
		// but set its score to 0 - the worst possible score
		best_individual=new Individual();
		best_individual->copy(individuals[0]);
		for(i=1;i<POPULATION_SIZE;i++){
			ammend_best_individual(individuals[i]);
		}
	
		// create a heap
		make_heap(individuals, individuals+POPULATION_SIZE,compare_individuals) ;

		print_statistics();
	}

	~Population(){
		for(unsigned i=0;i<POPULATION_SIZE;i++){
			delete individuals[i];
			delete next_generation[i];
		}
		delete individuals;
		delete next_generation;
		delete print_buffer;

		delete best_individual;
	}

	void insert(){
		current_placeholder->number=++total;
		counter++;

		ammend_best_individual(current_placeholder);

		if(counter<POPULATION_SIZE){
			current_placeholder=next_generation[counter];
		}
		else if(counter==POPULATION_SIZE){
			// the array is full now, create a heap
			make_heap(next_generation, next_generation+POPULATION_SIZE,compare_individuals) ;

			current_placeholder=next_generation[0];
		}
		else{
			// the front element has been modified; restore the heap

			// put the front element to the back of the heap, and restore the heap
			pop_heap(next_generation, next_generation+POPULATION_SIZE,compare_individuals);
			// add the last element to the heap
			push_heap(next_generation, next_generation+POPULATION_SIZE,compare_individuals);

			current_placeholder=next_generation[0];
		}
	}

	void step(){
		unsigned i,j,fitness;
		Individual *ind;

		improvement=false;
		generation_number++;

		// compute the population's average score
		double average_score=0;
		for(i=0;i<POPULATION_SIZE;i++){
			average_score+=individuals[i]->score;
		}
		average_score/=POPULATION_SIZE;

		// compute the next generation
		counter=0;
		current_placeholder=next_generation[0];

		for(i=0;i<POPULATION_SIZE;i++){
			ind=individuals[i];

			// compute the "fitness" of the individual - a parameter
			// regulating how many descendents it leaves - as follows:
			// 1) if the score of the individ is less then the average 
			//    score then fitness=1
			// 2) if the score of the individ is greater then the average 
			//    score then fitness=max(3,score-average_score)

			if(ind->score<average_score) fitness=1;
			else{
				fitness=ind->score-unsigned(average_score);
				if(fitness<3) fitness=3;
			}

			// individual spawns mutated ones, according to its fitness
			for(j=0;j<fitness;j++){
				current_placeholder->mutate(ind);
				score_hook(current_placeholder);
				insert();
			}

			// as every individual leaves at least one offspring, the
			// extinction is impossible
		}

		Individual** buf=next_generation;
		next_generation=individuals;
		individuals=buf;

		if(improvement) print_statistics();
	}

	// calls step() n times
	void steps(unsigned n){
		for(unsigned i=0;i<n;i++){
			if(best_individual->score>=max_score) break;
			step();
		}
	}

	void print_statistics(){
		printf("\n\ngeneration=%ld, fitness=%d, total=%ld",
			generation_number,
			best_individual->score,
			total
		);

		unsigned i,counter=0;
		// find the individuals having the highest score
		// and store them in print_buffer 
		for(i=0;i<POPULATION_SIZE;i++){
			if(individuals[i]->score==best_individual->score){
				print_buffer[counter++]=individuals[i];
			}
		}

		// sort print_buffer according to the individuals' numbers
		qsort(print_buffer,counter,sizeof(Individual*),compare_individuals_num);

		for(i=0;i<counter;i++){
			printf("\n");
			print_hook(print_buffer[i]);
		}
		fflush(stdout);
	}
};

#endif