#ifndef __MOEAD_H_
#define __MOEAD_H_

#include "global.h"
#include "common.h"
#include "individual.h"
#include "scalarfunc.h"
#include "recombination.h"

class TMOEAD    
{
public:

	TMOEAD();
	virtual ~TMOEAD();

	void init_uniformweight(int sd);    // initialize the weights for subproblems
	void init_neighbourhood();          // calculate the neighbourhood of each subproblem
	void init_population();             // initialize the population
	void standard_lattice_design(TSOP &sop, int &more, int h);

	void update_soperence(TIndividual &ind);           // update the approximation of ideal point
	void update_nadirpoint();						   // update the approximation of nadir point
	void update_problem(TIndividual &child, int id);   // compare and update the neighboring solutions
	void update_problem_global(TIndividual &child, int id);
	void normalize_objective(TIndividual &ind);

	void evolution();                                  // mating restriction, recombination, mutation, update
	void matingselection(vector<int> &list, int cid, int size, int type);
	void run(int sd, int nc, int mg, int rn);          // execute MOEAD
	void save_front(char savefilename[1024]);          // save the pareto front into files

    vector <TSOP>  population;  // current population  


	TIndividual *indivpoint;    // soperence point
	int  niche;                 // neighborhood size
	int  pops;                  // population   size	

	void operator=(const TMOEAD &emo);
};

TMOEAD::TMOEAD()
{

	idealpoint = new double[numObjectives];
	indivpoint = new TIndividual[numObjectives]; 

	nadirpoint = new double[numObjectives];
	// initialize ideal point	
    for(int n=0; n<numObjectives; n++) 
	{
		idealpoint[n] = 1.0e+30;  
		indivpoint[n].rnd_init();
		indivpoint[n].obj_eval();
	}
}

TMOEAD::~TMOEAD()
{
	delete [] idealpoint;    
	delete [] indivpoint;
	delete [] nadirpoint;
}


void TMOEAD::init_population()
{
    for(int i=0; i<pops; i++)
	{
		population[i].indiv.rnd_init();
		population[i].indiv.obj_eval();
		update_soperence(population[i].indiv);
	}
}

void TMOEAD::standard_lattice_design(TSOP &sop, int &more, int h)
{
	int i, j;
	if (!more)
	{
		more = 1;
		j = 1;
		vector <int> vlist(numObjectives, 0);
		vlist[0] = h;
		sop.array.assign(&vlist[0], &vlist[0] + numObjectives);
		for (i = 0; i<numObjectives; i++)
			sop.namda.push_back(1.0*vlist[i] / h);
	}
	else
	{
		j = numObjectives - 1;
		for (i = numObjectives - 2; i >= 0; i--)
		{
			if (sop.array[i]>0)
			{
				j = i;
				break;
			}
		}
		sop.array[j] -= 1;
		sop.array[j + 1] = h;
		for (i = 0; i <= j; i++)
			sop.array[j + 1] -= sop.array[i];
		for (i = j + 2; i<numObjectives; i++)
			sop.array[i] = 0;

		for (i = 0; i<sop.array.size(); i++)
			sop.namda[i] = 1.0*sop.array[i] / h;
		if (sop.array[numObjectives - 1] == h)
			more = 0;
	}
	return;
}

// initialize a set of evely-distributed weight vectors
void TMOEAD::init_uniformweight(int sd)
{   
	/********systematic approach******************/
	
	for (int i = 0; i <= sd; i++)
	{
		if (numObjectives == 2)
		{
			TSOP sop;
			sop.array.push_back(i);
			sop.array.push_back(sd - i);
			for (int j = 0; j < sop.array.size(); j++)
				sop.namda.push_back(1.0*sop.array[j] / sd);
			sop.array.clear();
			population.push_back(sop);
		}
		else if (numObjectives == 3)
		{
			for (int j = 0; j <= sd; j++)
			{
				if (i + j <= sd)
				{
					TSOP sop;
					sop.array.push_back(i);
					sop.array.push_back(j);
					sop.array.push_back(sd - i - j);
					for (int k = 0; k < sop.array.size(); k++)
						sop.namda.push_back(1.0*sop.array[k] / sd);
					sop.array.clear();
					population.push_back(sop);
				}
			}
		}
	}
	if (numObjectives >3)
	{
		TSOP sop;
		int more = 0;
		int unit = objDiv[0];

		while (1)
		{
			standard_lattice_design(sop, more, unit);
			sop.array.clear();
			population.push_back(sop);
			if (!more) break;
		}
		
		TSOP sop1;
		more = 0;
		unit = objDiv[1];
		while (unit)
		{				
			standard_lattice_design(sop1, more, unit);
			for (int i = 0; i < numObjectives; i++)
				sop1.namda[i] = 0.5*unit / numObjectives + 0.5*sop1.namda[i];
			sop1.array.clear();
			population.push_back(sop1);
			if (!more) break;
		}
	}
	pops = population.size();

	/************k-layer method***************/
	/*int klayer=sd;
	double b, a1, b1, centre=1.0/numObjectives;
	if(numObjectives==2)
	{
		for(int i=0; i<=sd; i++)
		{
			Tsop sop;		   
			sop.array.push_back(i);
			sop.array.push_back(sd-i);
			for(int j=0; j<sop.array.size(); j++)
				sop.namda.push_back(1.0*sop.array[j]/sd);
			population.push_back(sop); 
		}
	}
	else
	{
		for (int i=1; i<=numObjectives; i++)
		{
			for (int j=1; j<=klayer; j++)
			{
				Tsop sop;
				for (int m=0; m<numObjectives; m++)
				{
					if (i==(m+1))
						b=1.0;
					else
						b=0.0;
					sop.namda.push_back(centre+j*(b-centre)/klayer);
				}
				population.push_back(sop);
			}
		}
		
		for (int i=1; i<=numObjectives; i++)
		{
			for (int j=1; j<=klayer; j++)
			{
				for (int t=1; t<=j;t++)
				{
					Tsop sop;
					for (int m=0; m<numObjectives; m++)
					{
						if (i==(m+1)) 
							b=1.0;
						else
							b=0;
						a1=centre+j*(b-centre)/klayer;
						if ((m==i)||((i==numObjectives)&&(m==0)))
							b=1.0;
						else
							b=0;
						b1=centre+j*(b-centre)/klayer;
						sop.namda.push_back(a1+t*(b1-a1)/(j+1));
					}
					population.push_back(sop);
				}
			}
		}
		Tsop sop;
		for (int m=0; m<numObjectives; m++)
			sop.namda.push_back(centre);
		population.push_back(sop);
	}
	pops = population.size();*/

	niche = (int) (0.1*pops);
}

// initialize the neighborhood of subproblems based on the distances of weight vectors
void TMOEAD::init_neighbourhood()
{
    double *x   = new double[pops];
	int    *idx = new int[pops];
	for(int i=0; i<pops; i++)
	{	
		for(int j=0; j<pops; j++)
		{
		    x[j]    = distanceVector(population[i].namda,population[j].namda);
			idx[j]  = j;			
		}
		minfastsort(x,idx,pops,niche);   
		for(int k=0; k<niche; k++)   			
			population[i].table.push_back(idx[k]);

	}

    delete [] x;
	delete [] idx;
}

// update the best solutions of neighboring subproblems
void TMOEAD::update_problem(TIndividual &indiv, int id)
{
	int *perm = new int[niche];
	random_permutation(perm, niche);

	int count = 0;
	int i;

	normalize_objective(indiv);
	for (i = 0; i < niche; i++)
	{
		int    k = population[id].table[perm[i]];
		normalize_objective(population[k].indiv);

		double f1, f2;
		f1 = scalar_func2(population[k].indiv.yn_obj, population[k].namda, population[k].theta, indivpoint);
		f2 = scalar_func2(indiv.yn_obj, population[k].namda, population[k].theta, indivpoint);
		
		//f1 = scalar_func(population[k].indiv.y_obj, population[k].namda, indivpoint);
		//f2 = scalar_func(indiv.y_obj, population[k].namda, indivpoint);
		if (f2 < f1)
		{
			population[k].indiv = indiv;
			count++;
		}
		if (count >= nr) { break; }
	}

	delete[] perm;
}

// update globaly for the most suitable subproblem
void TMOEAD::update_problem_global(TIndividual &indiv, int csize)
{
	double *x = new double[pops];
	int    *idx = new int[pops];

	normalize_objective(indiv);
	for (int i = 0; i < pops; i++)
	{
		idx[i] = i;
		x[i] = scalar_func2(indiv.yn_obj, population[i].namda, population[i].theta, indivpoint);

	}
	minfastsort(x, idx, pops, niche);
	//update_problem(indiv, idx[0]);

	double count = 0;
	int j;
	for (j = 0; j < niche; j++)
	{
		normalize_objective(population[idx[j]].indiv);
		double fj = scalar_func2(population[idx[j]].indiv.yn_obj, population[idx[j]].namda, population[idx[j] ].theta, indivpoint);
		if (fj>x[j])
		{
			population[idx[j]].indiv = indiv;
			count++;
		}
		if (count >= nr) 
		{
			break;
		}
	}
	delete[] x;
	delete[] idx;

}

// update the reference point
void TMOEAD::update_soperence(TIndividual &ind)
{
	for(int n=0; n<numObjectives; n++)    
	{
		if(ind.y_obj[n]<idealpoint[n])
		{
			idealpoint[n]  = ind.y_obj[n];
			indivpoint[n]  = ind;
		}		

	}
}

// update the nadir point
void TMOEAD::update_nadirpoint()
{
	for (int n = 0; n < numObjectives; n++)
	{
		nadirpoint[n] = -INFINITY;
		for (int i = 0; i < pops; i++)
		{
			if (population[i].indiv.y_obj[n]>nadirpoint[n])
			{
				nadirpoint[n] = population[i].indiv.y_obj[n];
			}
		}
	}
}

void TMOEAD::normalize_objective(TIndividual &ind)
{
	for (int n = 0; n < numObjectives; n++)
		ind.yn_obj[n] = (ind.y_obj[n] - idealpoint[n]) / (nadirpoint[n] - idealpoint[n]);
}

void TMOEAD::matingselection(vector<int> &list, int cid, int size, int type){
	// list : the set of the indexes of selected mating parents
	// cid  : the id of current subproblem
	// size : the number of selected mating parents
	// type : 1 - neighborhood; otherwise - whole population
	int ss = population[cid].table.size(), r, p;
	while (list.size()<size)
	{
		if (type == 1){
			r = int(ss*rnd_uni(&rnd_uni_init));
			p = population[cid].table[r];
		}
		else
			p = int(population.size()*rnd_uni(&rnd_uni_init));

		bool flag = true;
		for (int i = 0; i<list.size(); i++)
		{
			if (list[i] == p) // p is in the list
			{
				flag = false;
				break;
			}
		}

		if (flag) list.push_back(p);
	}
}

// recombination, mutation, update in MOEA/D
void TMOEAD::evolution()
{
	int *perm = new int[pops];
	random_permutation(perm, pops);
    for(int i=0; i<pops; i++)
	{
		int n = perm[i];
		int type;
		double rnd = rnd_uni(&rnd_uni_init);

		// mating selection based on probability
		if (rnd<realb)    type = 1;   // neighborhood
		else              type = 2;   // whole population
		//type = 1;

		// select the indexes of mating parents
		vector<int> p;
		matingselection(p, n, 2, type);  // neighborhood selection

		TIndividual child, child2;
		// diff_evo_xover2(population[n].indiv,population[p[0]].indiv,population[p[1]].indiv,child);
		//realbinarycrossover(population[p[0] ].indiv,population[p[1] ].indiv,child, child2);  
		//realmutation(child, 1.0/numVariables);


		LLcrossover(population[p[0] ].indiv, population[p[1] ].indiv, child);
		LLmutation(child, 1.0 / numVariables);
		child.obj_eval();
		update_soperence(child);
		//update_problem(child, n);
		update_problem_global(child, i);
	}
	delete[]  perm;
}

void TMOEAD::run(int sd, int nc, int mg, int rn)
{
    // sd: integer number for generating weight vectors
	// nc: size of neighborhood
	// mg: maximal number of generations 
	//niche = nc;	
	init_uniformweight(sd);
    init_neighbourhood();
	init_population();	
	char savefilename[1024];
	char savepara[1024];
	sprintf(savepara, "%s\\para.txt", fileName);
	//std::fstream fout;
	//fout.open(savepara, std::ios::out);
	update_nadirpoint();
	for(gen=1; gen<=mg; gen++)   
	{
		evolution();
		update_nadirpoint();
		if ((gen < 10)||(gen <= 100 && gen % 10 == 0) || (gen <= 1000 && gen % 100 == 0) || (gen <= 10000 && gen % 1000 == 0))
		{
			savefilename[0] = '\0';
			sprintf(savefilename, "%s/pop_gen_%i.txt", fileName, gen);
			save_front(savefilename);

			//cout << "sr=%f" << 1.0*rs / (gen*pops) << " \t #comparisons=%f " << 1.0*tr / (niche*gen*pops) << endl;
			//fout << 1.0*rs / (gen*pops) << "\t" << 1.0*tr / (niche*gen*pops)<<endl;
		}

		cout << "gen= " << gen << endl;
	}
	//fout.close();
	savefilename[0] = '\0';
	sprintf(savefilename, "%s/lastpop.txt", fileName);
	save_front(savefilename);
	population.clear();
	
}


void TMOEAD::save_front(char saveFilename[1024])
{
    std::fstream fout;
    fout.open(saveFilename,std::ios::out);


    for(int n=0; n<population.size(); n++)
    {
            for(int k=0;k<numObjectives;k++)
                    fout<<population[n].indiv.y_obj[k]<<"  ";
            fout<<"\n";
    }

    fout.close();
}


void TMOEAD::operator=(const TMOEAD &emo)
{
    pops        = emo.pops;
	population  = emo.population;
	indivpoint  = emo.indivpoint;
	niche       = emo.niche;
} 


#endif