#ifndef __SPEAR_H_
#define __SPEAR_H_

#include "global.h"
#include "recombination.h"
#include "common.h"
#include "individual.h"

using namespace std;
class SPEAR
{
public:
	SPEAR();
	virtual ~SPEAR();

	void population_initialize();					// population initialization
	void refset_initialize();						// reference set initialization
	void population_merge();						// parent and offspring population combination
	void update_idealpoint(vector <double> &objV);  // ideal point update
	void find_worstobj(bool nondominate, bool ASFon);
	void create_hyperplane();
	void member_assoicate();						// population and reference set association
	void fitness_assignment();						// fitness assignment
	void environmental_selection();					// environmental selection
	int restricted_mating(int id, int K);			// restricted mating
	void reproduction();							// population reproduction
	void spear_run(int run);						// execution of SPEA/R

	double find_maxdelta();							// identification of maximal angles between reference directions

	void SLD_subfunction(REF &ref, int &more, int h);//subfunctions in SLD
	void SLD_design(vector <REF> &refset, int h);	// standard simplex lattice design
	void klayer_design();							// k-layer design

	void save_nds(char savefilename[1024]);         // save the nomdominated front set into files
	void save_pof(char savefilename[1024]);         // save the pareto front into files
	void save_pos(char savefilename[1024]);		    // save the pareto set into files

	vector <IND>       population;					// evolving population
	vector <IND>       merge_pop;					// merged population
	vector <REF>       refset;						// reference direction set
	vector <IND>	Nondom;							// nondominated set
	vector <double> idealpoint;						// ideal point
	vector <double> extremepoint;					// extremepoint
	vector <int>    extremearray;					// array of extreme points

	int                 mergesize;					// the size of merged population
	double 				maxdelta;					// the maximum acute angle between each pair of neighbouring ref directions.
	int *perm;										// global variable for permution

};


// create SPEAR instance
SPEAR::SPEAR()
{
	// initialize ideal point and extreme point
	idealpoint.assign(n_obj, INF);
	extremepoint.assign(n_obj, -INF);
	extremearray.assign(n_obj,-1);
	perm = new int[popsize];
}

// destroy SPEAR instance
SPEAR::~SPEAR()
{
	idealpoint.clear();
	extremepoint.clear();
	extremearray.clear();

	population.clear();
	merge_pop.clear();
	refset.clear();

	delete[] perm;
}

//Simplex lattice design (SLD)
void SPEAR::SLD_design(vector <REF> &refset, int h)
{
//==================================simplex-lattice design==========================================////
	int more = 0, k = n_obj - 1;																	 //
	while (nchoosek(++k + n_obj - 1, n_obj - 1) <= popsize){}// calculate objective division value	 //
																									 //
	unit = k - 1;																					 //
	REF ref;																						 //
	while (1)																						 //
	{																								 //
		SLD_subfunction(ref, more,h);															     //
		refset.push_back(ref);																		 //
		if (!more) break;																			 //
	}																								 //
//==================================================================================================////
}

// subfuntions in Simplex lattice design (SLD)
void SPEAR::SLD_subfunction(REF &ref, int &more, int h)
{ // this a recursive way to generate reference directions
	int i, j;
	if (!more)
	{
		more = 1;
		j = 1;
		vector <int> vlist(n_obj, 0);
		vlist[0] = h;
		ref.array.assign(&vlist[0], &vlist[0] + n_obj);
		for (i = 0; i<n_obj; i++)
			ref.namda.push_back(1.0*vlist[i] / h);
	}
	else
	{
		j = n_obj - 1;
		for (i = n_obj - 2; i >= 0; i--)
		{
			if (ref.array[i]>0)
			{
				j = i;
				break;
			}
		}
		ref.array[j]  -=1;
		ref.array[j + 1] = h;
		for (i = 0; i <= j; i++)
			ref.array[j + 1] -=ref.array[i];
		for (i = j + 2; i<n_obj; i++)
			ref.array[i] = 0;

		for (i = 0; i<ref.array.size(); i++)
			ref.namda[i] = 1.0*ref.array[i] / h;
		if (ref.array[n_obj - 1] == h)
			more = 0;
	}
	return;
}

// k-layer approach to create reference direction set
void SPEAR::klayer_design()
{
	int k=0;
	double centre=1.0/n_obj, b=0.0;
	double a1, b1;
	// create points on the sides of subsimplexes 
	for (int i=1;i<=n_obj; i++)
		{
			for (int j=1; j<=klayer; j++)
			{
				for (int m=0; m<n_obj; m++)
				{
					if (i==(m+1)) 
						b=1.0;
					else
						b=0;

					refset[k].namda.push_back(centre+j*(b-centre)/klayer);

				}
				k++;
			}
		}

	// create points on one the line segments of subsimplexes
	for (int i=1; i<=n_obj; i++)
	{
		for (int j=1; j<=klayer; j++)
		{
			for (int t=1; t<=j;t++)
			{
				for (int m=0; m<n_obj; m++)
				{
					if (i==(m+1)) 
						b=1.0;
					else
						b=0;
					a1=centre+j*(b-centre)/klayer;
					if ((m==i)||((i==n_obj)&&(m==0)))
						b=1.0;
					else
						b=0;
					b1=centre+j*(b-centre)/klayer;
					refset[k].namda.push_back(a1+t*(b1-a1)/(j+1));
				}
				k++;
			}

		}
	}
	// include central points in the reference set
	for (int m=0; m<n_obj; m++)
		refset[k].namda.push_back(centre);
}

// reference set initialization
void SPEAR::refset_initialize()
{
	if(n_obj==2) // in the case of 2 objectives
	{
		SLD_design(refset, popsize-1);
	}
	else // in the case of more than 2 objectives
	{	 //==================================simplex-lattice design==========================================////
		if (!strcmp(optrefset, "SLD"))        																 //
		{																									 //
			int i = n_obj-1;																				 //	
			while (nchoosek(++i + n_obj - 1, n_obj - 1) <= popsize){} 										 //
			SLD_design(refset, i-1);																		 //
		}																									 //
		//===================================================================================================////

		//=================================(multiple) two-layered approach in NSGA-III=======================////
		else if (!strcmp(optrefset, "2SLD"))																 //
		{																									 //
			// please carefully configure the divs vector used in multiple layers 							 //
			int i = 0;																						 //
			while (refset.size() + nchoosek(divs[i] + n_obj - 1, n_obj - 1) <= popsize)						 //
			{ 																								 //
				SLD_design(refset, divs[i]); 																 //
				i++; 																						 //
			}																								 //
		}																									 //
		//===================================================================================================////

		//===============================K-layer (subsimplex design)=========================================////
		else if (!strcmp(optrefset,"KL")) 	// K-layer design												 //
		{																									 //
			int k = 1;																						 //
			while (sizelayer(n_obj, k) <= popsize) // calculate the number of layers						 //
				k++;																						 //
			klayer = k - 1;																					 //
			refset.resize(sizelayer(n_obj, klayer));														 //
			klayer_design();																				 //
		}																									 //
		//===================================================================================================////
	}
	refsize = refset.size();
	population.reserve(popsize);
	merge_pop.reserve(2*popsize);
}

// calculate the maximal acute angle between reference vectors
double SPEAR::find_maxdelta()
{
	double maxvalue=-INF;
	//-----delta=max(min(<w_i, w_j>))----/
	for(int i=0; i<refsize; i++)
	{
		double minvalue=INF;
		// calculate the distances based on reference vectors
		for(int j=0; j<refsize; j++)
		{
			if (j != i)
			{
				double tmp = angleVector(refset[i].namda, refset[j].namda);
				if (tmp < minvalue)
					minvalue = tmp;
			}
		}
		if (maxvalue<minvalue)
			maxvalue=minvalue;
	}
	return maxvalue;
}

// population initialization
void SPEAR::population_initialize()
{
	for (int i=0; i<popsize; i++)
	{
		IND ind;
		ind.rnd_init();
		ind.evaluation();
		update_idealpoint(ind.y_obj);
		population.push_back(ind);		
	}
	return;
}

// population combination
void SPEAR::population_merge()
{
	bool flag, checkrpt;
	checkrpt = true; // false - for efficiency, switch off check repeats 

	if (checkrpt)// check repeats
	{
		for (int i = 0; i < population.size(); i++)
		{
			bool flag = true;
			for (int j = 0; j < merge_pop.size(); j++) // inefficient part for checking repeats.
			{
				if (population[i] == merge_pop[j])
				{
					flag = false;
					break;
				}
			}
			if (flag)  merge_pop.push_back(population[i]);
		}
	}
	else // do not check repeats
	{
		for (int i = 0; i < population.size(); i++)
			merge_pop.push_back(population[i]);
	}

	mergesize = merge_pop.size();
	
	// translate ind objective vector 
	for (int i = 0; i < mergesize; i++)
		minusVector(merge_pop[i].y_tobj, merge_pop[i].y_obj, idealpoint);
	return;
}

// identification of extreme points/intercepts
void SPEAR::find_worstobj(bool nondominate, bool ASFon)
{
	vector<double> w;
	int index = -1;
	
	for (int k = 0; k < n_obj; k++)
	{
		double min = INF;
		w.assign(n_obj, 10e-6);
		w[k] = 1.0;

		for (int i = 0; i < mergesize; i++)
		{
			if (nondominate && (merge_pop[i].fitness < 1.0)) // actions on nondominated set
			{
				if (ASFon) // find extreme points by ASF
				{
					double asf = ASF(merge_pop[i].y_tobj, w);
					if (min>asf)
					{
						min = asf;
						index = i;
					}
				}
				else // find extreme points the worst objective value
				{
					if (extremepoint[k] < merge_pop[i].y_obj[k])
						extremepoint[k] = merge_pop[i].y_obj[k];
				}
			}
			else if (!nondominate) //actions on the whole merged population
			{
				if (ASFon) // find extreme points by ASF
				{
					double asf = ASF(merge_pop[i].y_tobj, w);
					if (min>asf)
					{
						min = asf;
						index = i;
					}
				}
				else // find extreme points the worst objective value
				{
					if (extremepoint[k] < merge_pop[i].y_obj[k])
						extremepoint[k] = merge_pop[i].y_obj[k];
				}
			}
		}
		extremearray[k] = index;

	}// end or loop

}

// create a hyperplane to calculate intercepts
void SPEAR::create_hyperplane()
{
	double duplicate = false, negative = false;

	for (int i = 0; (!duplicate && i < n_obj); i++)
		for (int j = i + 1; (!duplicate && i < n_obj); j++)
			duplicate = (extremearray[i] == extremearray[j]);

	if (!duplicate)
	{
		vector< vector<double> > A;
		for (int i = 0; i < n_obj; i++)
			A.push_back(merge_pop[extremearray[i]].y_tobj);

		negative = calc_intercept(extremepoint, A);
	}

	if (duplicate || negative)
		find_worstobj(false, false);

}

// population association
void SPEAR::member_assoicate() 
{
	// clear the member table for each reference direction.
	for (int i = 0; i<refsize; i++)
	{
		refset[i].member.clear();
	}

	double d, min ;
	int mark;

	// associate each member to a reference direction
	for (int i = 0; i<mergesize; i++)
	{
		min = INF;
		for (int n = 0; n < n_obj; n++)
			 merge_pop[i].y_tobj[n] /= (extremepoint[n] - idealpoint[n]);
		for (int j = 0; j<refsize; j++)
		{
			d = angleVector(merge_pop[i].y_tobj, refset[j].namda);
			if (d < min)
			{
				mark = j;
				min = d;
			}
		}
		merge_pop[i].refno=mark;
		merge_pop[i].angvalue=min;
		refset[mark].member.push_back(i);
	}
}

// ideal point update
void SPEAR::update_idealpoint(vector <double> &objV)
{
	for (int n = 0; n < n_obj; n++)
	{
		if (objV[n] < idealpoint[n])
			idealpoint[n] = objV[n];
	}
}

void SPEAR::fitness_assignment() // most time-consuming part
{
	int *gstrength = new int[mergesize];
	int *lstrength = new int[mergesize];
	double *lfitness = new double[mergesize];

	int **dominate = new int*[mergesize];

	// variables initialization
	for (int i = 0; i < mergesize; i++)
	{
		gstrength[i] = 0;
		lstrength[i] = 0;
		lfitness[i] = 0.0;
		dominate[i] = new int[mergesize];
	}

	// calculate global raw fitness.
	for (int i = 0; i<mergesize; i++)
	{
		dominate[i][i] = 0;
		for (int j = i + 1; j < mergesize; j++)
		{
			dominate[i][j] = merge_pop[i].check_dominate(merge_pop[j]);
			dominate[j][i] = -dominate[i][j];
		}

		gstrength[i] = sum_array_match(dominate[i], 1, mergesize);	
	}

	// calculate global fitness.	
	for (int i = 0; i<mergesize; i++)
		merge_pop[i].fitness = sum_array_match(dominate[i], gstrength, -1, mergesize);

	//======================= find extremepoints of approximated POF=================================////
	extremepoint.assign(n_obj, -INF);																 //
	switch (ExtreSet)																				 //
	{																								 //
		case 0: find_worstobj(true, true); create_hyperplane();  break;								 //
		case 1: find_worstobj(false, true); create_hyperplane();  break;							 //
		case 2: find_worstobj(true, false); break;													 //
		case 3: find_worstobj(false, false); break;													 //
		default:  break;																			 //
	}																								 //
																									 //
	//===============================================================================================////
	// associate population members with refset.
	member_assoicate();

	//====================== calculate local strength and raw fitness================================////	
	for (int i = 0; i<refsize; i++)																	 //
	{																								 //
		int memsize = refset[i].member.size();														 //
		for (int j = 0; j<memsize; j++)																 //
		{																							 //
			for (int k = 0; k<memsize; k++)															 //
			{																						 //
				int jd = refset[i].member[j];														 //
				int kd = refset[i].member[k];														 //
																									 //
				if (dominate[jd][kd] == 1)															 //
					lstrength[jd] += 1;																 //
			}																						 //
		}																							 //
		for (int j = 0; j<memsize; j++)																 //
		{																							 //
			for (int k = 0; k < memsize; k++)														 //
			{																						 //
				int jd = refset[i].member[j];														 //
				int kd = refset[i].member[k];														 //
																									 //
				if (dominate[jd][kd] == -1)															 //
					lfitness[jd] += 1.0*lstrength[kd];												 //
			}																						 //
		}																							 //
	}// end for																						 //
	//===============================================================================================////

	//=============================== calculate each member's final fitnes===========================////
	for (int i = 0; i<mergesize; i++)																 //
	{																								 //
		// calculate density information															 //
		double angle = merge_pop[i].angvalue / (maxdelta + merge_pop[i].angvalue);					 //
		int k = merge_pop[i].refno;																	 //
																									 //
		if ((refset[k].member.size() == 1) && (gen<max_gen)) 										 //
			// if only a single dominated member in the subregion and evolution not due				 //
			merge_pop[i].fitness = lfitness[i] + angle;   // do not use global fitness				 //
		else																						 //
			merge_pop[i].fitness += lfitness[i] + angle;	// use both global and local fitness	 //
	}																								 //
	//===============================================================================================////

	for (int i = 0; i < mergesize; i++)
		delete[] dominate[i];

	delete[] lstrength;
	delete[] gstrength;
	delete[] lfitness;
	delete[] dominate;
}


// environmental selection
void SPEAR::environmental_selection()
{
	int psize = 0, tmpsize = 0;
	double min;
	vector <int> record(mergesize,1);  // indicate member has been selected or not
	int *tmpmember=new int[mergesize]; // store selected members
	
	//================= diversity-first-convergence-second selection based on reference directions===================////
	while (psize < popsize)																							 //
	{																												 //
		tmpsize = 0;																								 //
		for (int j = 0; j < refsize; j++) 																			 //
		{																											 //
			min = INF;																								 //
			int id = -1;																							 //
			for (int i = 0; i < refset[j].member.size(); i++) 														 //
			{																										 //
				int t = refset[j].member[i];																		 //
				if (record[t])																						 //
				{																									 //
					if (min > merge_pop[t].fitness) 																 //
					{																								 //
						min = merge_pop[t].fitness;																	 //
						id = t;																						 //
					}																								 //
				}																									 //
			}																										 //
			if (id >= 0) 																							 //
			{																										 //
				tmpmember[tmpsize++] = id;																			 //
			}																										 //
		}																											 //
		if (psize + tmpsize <= popsize)																				 //
		{// selected inds cannot fill up the population																 //
			for (int i = 0; i < tmpsize; i++)																		 //
			{																										 //
				if (gen < max_gen || (merge_pop[ tmpmember[i] ].fitness < 1.0 && (gen == max_gen)))					 //
				{																									 //
					population[psize++] = merge_pop[ tmpmember[i] ];												 //
					record[ tmpmember[i] ] = 0;																		 //
				}																									 //
			}																										 //
			if (psize == popsize) break;																			 //
		}																											 //
		else																										 //
		{// selected inds exceed the remaining slots																 //
			for (int i = 0; i < tmpsize; i++) 																		 //
			{																										 //
				double min = merge_pop[ tmpmember[i] ].fitness;														 //
				int mark = i;																						 //
				for (int j = i + 1; j < tmpsize; j++) 																 //
				{																									 //
					if (min > merge_pop[ tmpmember[j] ].fitness) 													 //
					{																								 //
						mark = j;																					 //
						min = merge_pop[ tmpmember[j] ].fitness;													 //
					}																								 //
				}																									 //
				int temp = tmpmember[i];																			 //
				tmpmember[i] = tmpmember[mark];																		 //
				tmpmember[mark] = temp;																				 //
																													 //
				population[psize++] = merge_pop[ tmpmember[i] ];													 //
				if (psize == popsize) break;																		 //
			}																										 //
		}																											 //
	}//end wihle																									 //
	//===============================================================================================================////
	merge_pop.assign(population.begin(), population.end());
	delete[] tmpmember;
}

// restricted mating
int SPEAR::restricted_mating(int id1,int K)
// return a parent with its index
{
	int returnID=-1;
	int r1=merge_pop[id1].refno;
	random_permutation(perm, popsize);

    double min = INF;
    int temp = 0;
    for (int i = 0; i < popsize; i++) 
	{
        int r2=merge_pop[perm[i]].refno;
        if ((r2 == r1)) continue;
        if (perm[i] != id1) 
		{
            temp++;
            int d = distanceVector(merge_pop[id1].y_tobj, merge_pop[perm[i]].y_tobj);
            if (d < min) 
			{
                min = d;
                returnID = perm[i];
            }
        }
        if (temp >= K) break;
    }
	if (returnID < 0)
	{
		temp = 0;
		while (perm[temp] == id1)
			temp++;
		returnID = perm[temp];

	}
	return returnID;
}

// population reproduction
void SPEAR::reproduction()
{	
	vector <IND> parent(2);
	vector <IND> child(2);
    for (int i = 0; i < popsize; i++) 
	{		
		parent[0]=merge_pop[i];
		int rp = restricted_mating(i,Ksize);
        parent[1]=merge_pop[rp];
        crossover (parent, child);
		mutation(child[0], 1.0 / n_var);
		child[0].evaluation();
		update_idealpoint(child[0].y_obj);
		population[i]=child[0]; 		
    }
}

void SPEAR::spear_run(int run)
{
	seed = (seed + 23)%1377;
	rnd_uni_init = -(long)seed;
	
	char filename[256];

	//======starting point===================
	refset_initialize();																
	maxdelta = find_maxdelta();															
	population_initialize();															
	population_merge();																	
	fitness_assignment();																
																						
	gen=1;																				
	while (gen<=max_gen)																
	//======generational evolution===========
	{
		cout << "gen = " << gen << endl;
		reproduction();
		population_merge();
		fitness_assignment();
		environmental_selection();

		//if (gen<11 || (gen <= 200 && gen % 10 == 0) || (gen <= max_gen && gen % 100 == 0))
		if (gen==max_gen)
		{
			//sprintf_s(filename, "%s\\pop_gen%i.txt", parName, gen);
			sprintf_s(filename,"%s\\lastpop.txt",parName);
			save_pof(filename);
		}
		gen++;
	}

	return;
}	

void SPEAR::save_pof(char saveFilename[1024])
{
	std::fstream fout;
	fout.open(saveFilename,std::ios::out);
	for(int n=0; n<popsize; n++)
	{
		for(int k=0;k<n_obj;k++)
			//fout << merge_pop[n].y_obj[k] << "  ";
			fout<<population[n].y_obj[k]<<"  ";
		fout<<"\n";
	}
	fout.close();
}

void SPEAR::save_nds(char saveFilename[1024])
{
	std::fstream fout;
	fout.open(saveFilename, std::ios::out);
	for (int n = 0; n<Nondom.size(); n++)
	{
		for (int k = 0; k<n_obj; k++)
			fout << Nondom[n].y_obj[k] << "  ";
		fout << "\n";
	}
	fout.close();
}

void SPEAR::save_pos(char saveFilename[1024])
{
	std::fstream fout;
	fout.open(saveFilename,std::ios::out);
	for(int n=0; n<popsize; n++)
	{
		for(int k=0;k<n_var;k++)
			fout<<population[n].x_var[k]<<"  ";
		fout<<"\n";
	}
	fout.close();
}


#endif