#ifndef __CSGEACLASS_H_
#define __CSGEACLASS_H_

#include "..\common\global.h"
#include "..\common\recombination.h"
#include "..\common\common.h"
#include "sgeaclass.h"


class CSGEA
{

public:
	CSGEA();
	virtual ~CSGEA();

	void init_population();                  // initialize the population


	vector <int> rank_pop(vector <CSGEAInd> &pop);        // identify nondominated set
	void update_population(CSGEAInd &indiv);              // update population 
	void update_archive(CSGEAInd &ind);                    // update archive

	void sbxevolution();                                  // SBX-based recombination
	int  tour_selection();								  // tournament selection of parents

	void environmentselection();						  // environmental selection
	void fill_union(CSGEAInd &ind);						  // population union
	vector  <int> truncation(vector <CSGEAInd> &pop, vector <int> &v1, int m);
	vector<vector <int> > preservation(vector <CSGEAInd> &pop, int m);

	void introduce_dynamics(int g);
	bool detect_change(int n);
	void react_change();
	void para_init();

	// execute SGEA
	void execute(int run);

	void read_front(char filename[1024]);
	void calc_distance();

	void save_front(char savefilename[1024]);       // save the pareto front into files
	void save_ps(char savefilename[1024]);

	vector <CFIN>       population;
	vector <CSGEAInd>   ps;
	vector <CSGEAInd>   offspring;
	vector <int>        array;

	vector <CSGEAInd>   Archive;
	int                 nondomsize;


	vector<double>  C0, C1;                         // Store centriod of POS in the past two POSs

	double              distance;                   // generational distance from PF to solutions found
	int                 popsize;
	int                 archsize;
	void operator=(const CSGEA &moea);
};

CSGEA::CSGEA()
{

}

CSGEA::~CSGEA()
{

}

// initializae population
void CSGEA::init_population()
{
	for (int i = 0; i < pops; i++)
	{
		CFIN sub;
		population.push_back(sub);
		population[i].indiv.rnd_init();
		population[i].indiv.obj_eval();

		nfes++;
	}
	detectors = (int) (0.1*pops); // 10% of population used for detection
}

// "=" operator on data of CSGEA 
void CSGEA::operator=(const CSGEA &alg)
{

	population = alg.population;
	ps = alg.ps;
	offspring = alg.offspring;
	distance = alg.distance;
	popsize = alg.popsize;
}

// update the evolving population whenever a new solution is created
void CSGEA::update_population(CSGEAInd &indiv)
{
	// indiv: child solution

	int k = 0, incomp = 0;
	double max = -1.0e+30;

	indiv.ncount = 0;


	for (int i = 0; i < population.size(); i++)
	{
		int dominate = population[i].indiv.compare(indiv);
		switch (dominate)
		{
		case 2: return;
			break;
		case 1: population[i].indiv.ncount++;
			break;
		case 0: incomp++;
			break;
		case -1: indiv.ncount++;
			break;
		default: break;
		}

		if (max < population[i].indiv.ncount)
		{
			k = i;
			max = population[i].indiv.ncount;
		}
	}

	// do nothing if indiv is worse than any ind in pop
	if (max < indiv.ncount) return;

	// replace a random ind with indiv if indiv and pop are nondominated
	if (incomp == population.size())
	{
		k = int(population.size()*rnd_uni(&rnd_uni_init));

	}

	indiv.fitness = 1.0*indiv.ncount;
	population[k].indiv = indiv;


	// update archive
	if (indiv.ncount < 1) update_archive(indiv);

}

// update archive whenever a new solution is created
void CSGEA::update_archive(CSGEAInd &ind)
{
	int size = Archive.size();
	for (int i = 0; i < size; i++)
	{
		int dominate = Archive[i].compare(ind);
		if (dominate == 2 || dominate == -1) break;
		if (dominate = 1)
		{
			Archive.erase(Archive.begin() + i);
			size--;
		}
	}
	Archive.push_back(ind);
}

// tournament selection of mating parents 
int  CSGEA::tour_selection()
{
	int p1 = int(rnd_uni(&rnd_uni_init)*population.size());
	int p2 = int(rnd_uni(&rnd_uni_init)*population.size());

	if (population[p1].indiv.fitness < population[p2].indiv.fitness)
		return p1;
	else
		return p2;
}

// population ranking : identification of nondominated solutions
vector  <int>  CSGEA::rank_pop(vector <CSGEAInd> &pop)
{
	// rank: only identify nondominated solutions in the population (0==nondominated)
	vector <int> nondom;

	for (int i = 0; i < pop.size(); i++)
	{
		pop[i].rank = 0;
	}

	int dominated;
	for (int i = 0; i < pop.size(); i++)
	{
		for (int j = 0; j < pop.size(); j++)
		{
			dominated = pop[j].compare(pop[i]);
			if (dominated == -1) pop[i].rank++;
		}
	}

	for (int i = 0; i < pop.size(); i++)
	{
		if (pop[i].rank < 1)
			nondom.push_back(i);
	}
	return nondom;
}

// environmental selection: maintain population in a generational manner
void CSGEA::environmentselection()
{
	// merge of current population and archive population
	// duplicate individuals are removed.

	for (int i = 0; i < population.size(); i++)
		fill_union(population[i].indiv);

	// identification of nondominated solutions
	Archive.clear();
	vector<vector<int> > Index;
	Index = preservation(offspring, population.size());
	vector< int> eliteIn(Index[0]);
	vector< int> nondomIn(Index[1]);
	nondomsize = nondomIn.size();
	if (nondomsize < population.size())
	{
		for (int i = 0; i < nondomsize; i++)
			Archive.push_back(offspring[nondomIn[i]]);
		for (int i = 0; i < population.size(); i++)
			population[i].indiv = offspring[eliteIn[i]];
	}
	if (nondomsize == population.size())
	{
		for (int i = 0; i<population.size(); i++)
		{
			population[i].indiv = offspring[nondomIn[i]];
			Archive.push_back(offspring[nondomIn[i]]);
		}
	}

	if (nondomsize>population.size())
	{
		// Reduce the risk of too many extreme solutions.
		if (nobj < 3) CES = 1.0E-4;
		else CES = 1.0E-3;
		vector <int> trunIn;
		trunIn = truncation(offspring, nondomIn, population.size());
		for (int i = 0; i < population.size(); i++)
		{
			population[i].indiv = offspring[trunIn[i]];
			Archive.push_back(offspring[trunIn[i]]);
		}
	}

	for (int i = 0; i<population.size(); i++)
		offspring[i] = population[i].indiv;
	while (offspring.size()>population.size()) offspring.pop_back();

}


// population preservation: identification of nondominated set and sorting of dominated solutions
vector  <vector <int> > CSGEA::preservation(vector <CSGEAInd> &pop, int m)
{

	double *fit = new double[pop.size()];
	int* idx2 = new int[pop.size()];

	vector <int> accept, reject;

	int id = 0;
	double tmp, max;

	// strength or fitness assignment
	for (int i = 0; i < pop.size(); i++)
	{
		pop[i].ncount = 0;

		for (int j = 0; j < pop.size(); j++)
		{
			if (i != j)
			{
				if (pop[i].compare(pop[j]) == 1)  pop[i].ncount++;
			}
		}

	}

	// nondominance identification
	vector <int> nondomindex;

	for (int i = 0; i < pop.size(); i++)
	{
		if (pop[i].ncount == 0)
		{
			nondomindex.push_back(i);
		}
		pop[i].fitness = 1.0*(pop[i].ncount);
		fit[i] = pop[i].fitness;
		idx2[i] = i;
	}

	minfastsort(fit, idx2, pop.size(), m);

	for (int i = 0; i < pop.size(); i++)
	{
		if (i < m) accept.push_back(idx2[i]);
	}

	delete[] fit;
	delete[] idx2;

	vector <vector <int> > twovector;
	twovector.push_back(accept);
	twovector.push_back(nondomindex);

	return twovector;
}

// population trunction in the case of excessive nondominated solutions
vector <int> CSGEA::truncation(vector <CSGEAInd> &pop, vector <int> &v1, int m)
{
	int size = v1.size();
	int **distIndex = new int*[size];
	double **distSort = new double*[size];

	vector <int> record(size, 1);
	vector <int> accept;

	// the k-th nearest neighbour truncation method
	double INF = 1.0E+30;
	for (int i = 0; i < size; i++)
	{
		distIndex[i] = new int[size];
		distSort[i] = new double[size];
		for (int j = 0; j < i; j++)
		{
			distIndex[i][j] = j;
			distSort[i][j] = dist_vector(pop[v1[i]].y_obj, pop[v1[j]].y_obj);
			distIndex[j][i] = i;
			distSort[j][i] = distSort[i][j];
		}
		distIndex[i][i] = i;
		distSort[i][i] = INF;
	}

	for (int i = 0; i < size; i++)
		minfastsort(distSort[i], distIndex[i], size, size);

	int cursize = size, mark, count;
	double min;
	bool same;
	while (cursize>m)
	{
		min = INF;
		for (int i = 0; i < size; i++)
			if (record[i])
			{
				if (min>distSort[i][0])
				{
					min = distSort[i][0];
					mark = i;
				}
				else
				{
					if (min == distSort[i][0])
					{
						same = (pop[v1[i]].y_obj == pop[v1[mark]].y_obj);
						if (same == false)
						{
							count = 0;
							while (distSort[mark][count] == distSort[i][count] && count < cursize - 1) count++;
							if (distSort[mark][count] > distSort[i][count]) mark = i;
						}
					}
				}
			}
		record[mark] = 0;
		for (int i = 0; i < size; i++)
		{
			int flag = 0;
			if (record[i])
			{
				for (int j = 0; j < cursize; j++)
				{
					if (distIndex[i][j] == mark) flag = 1;
					if (flag)
						if (j < cursize - 1)
						{
							distSort[i][j] = distSort[i][j + 1];
							distIndex[i][j] = distIndex[i][j + 1];
						}
				}
				distSort[i][cursize - 1] = INF;
				distIndex[i][cursize - 1] = mark;
			}
		}
		cursize--;
	}
	for (int i = 0; i < size; i++)
	{
		if (record[i]) accept.push_back(v1[i]);
		delete[] distSort[i];
		delete[] distIndex[i];
	}
	delete[] distSort;
	delete[] distIndex;

	return accept;

}

// population union
void CSGEA::fill_union(CSGEAInd &ind)
{
	bool flag = true;
	int  size = offspring.size();
	for (int i = 0; i < size; i++){
		if (ind == offspring[i]){
			flag = false;
			break;
		}
	}
	if (flag) offspring.push_back(ind);
}

// SBX-based evolution procedure
void CSGEA::sbxevolution()
{
	pops = population.size();
	int *perm = new int[pops];
	random_permutation(perm, pops);

	bool changeIsTrue = false, reactIsTrue = false;
	int times = 0;
	for (int i = 0; i < pops; i++)
	{
		int n = perm[i];

		// steady-state change detection and response
		if ((!changeIsTrue)&&(times<detectors))
		{
			changeIsTrue = detect_change(n);
			times++;
		}
		if ((!reactIsTrue) && changeIsTrue)
		{
			cout << "A change occurs at generation " << itt << endl;
			react_change();
			reactIsTrue = true;
		}

		// select the indexes of mating parents
		vector<int> p;
		CSGEAInd child1, child2;

		// mating selection from either the current population or the archive
		if (rnd_uni(&rnd_uni_init) < 0.5)
		{
			int p1, p2;
			p1 = (int)(Archive.size())*rnd_uni(&rnd_uni_init);
			while (1){
				p2 = tour_selection();
				int dominate = population[p2].indiv.compare(Archive[p1]);
				if (dominate != 2) break;
			}
			real_sbx_xover1(Archive[p1], population[p2].indiv, child1, child2);
		}
		else
		{
			int p1, p2;
			p1 = tour_selection();
			while (1){ p2 = tour_selection();  	if (p2 != p1) break; }
			real_sbx_xover1(population[p1].indiv, population[p2].indiv, child1, child2);
		}

		// apply polynomial mutation
		realmutation(child1, 1.0 / nvar);

		// evaluate the child solution
		child1.obj_eval();

		// update the population, and the archive 
		update_population(child1);
	}
	delete[] perm;

}

void CSGEA::execute(int run)
{
	para_init();

	seed = (seed + 23) % 1377;
	rnd_uni_init = -(long)seed;

	vector<double> gd;
	char filename[1024];

	// initialization 
	int gen = 0;
	nfes = 0;

	introduce_dynamics(gen);
	init_population();
	environmentselection();
	// evolution
	for (gen = 1; gen <= max_gen; gen++)
	{
		itt = gen;
		introduce_dynamics(gen);

		sbxevolution();
		environmentselection();

		//if((gen%max_gen==0))
		{
			sprintf_s(filename, "%s\\pop_gen_%i.txt", parName, gen);
			save_front(filename);
		}
	}

	population.clear();
}

// initialization of parameters related to dynamics
void CSGEA::para_init()
{
	CES = 1.0E-300;
	Parar = 0;
	Tt = 0.0;
}

// introduce dynamics to the evolution process
void CSGEA::introduce_dynamics(int gen)
{
	int g = (gen - T0) > 0 ? (gen - T0) : 0;
	if (g > 0)
		Tt = 1.0 / nt*(floor(1.0*(g - 1) / taut) + 1);
	else
	{
		Tt = 0.0;
		rnd_rt = 0.0;
	}

	if ((Tt > 0) && (g%taut == 1)){
		rnd_rt = rnd_uni(&rnd_uni_init);

		if (!strcmp(strTestInstance, "dMOP3"))
		{
			Parar = (int)nvar*rnd_uni(&rnd_uni_init);
		}
	}

	if (!strcmp(strTestInstance, "SDP1"))
	{
		if (Tt <= 0) // at least run one at the beginning
		{
			for (int i = nobj - 1; i < nvar; i++)
			{
				sdp_y[i] =1.0*(i + 1) / nvar;
			}
		}
		else if (g%taut == 1)
		{
			for (int i = nobj - 1; i < nvar; i++)
			{
				rnd_rt = rnd_uni(&rnd_uni_init);
				sdp_y[i] += 2*(rnd_rt - 0.5)*sin(0.5*pi*Tt);
				if ((sdp_y[i]>1.0) || (sdp_y[i] < 0.0))
					sdp_y[i] = rnd_rt;
			}

		}
	}

	if (!strcmp(strTestInstance, "SDP13"))
	{
		if (g%taut == 1)
		{
			int tmp = g / taut;
			if (tmp % 3 == 0)
				nobj = 2;
			else
				nobj = 2; //----------------
		}
	}
}

// change detection
bool CSGEA::detect_change(int n)
{
	// classic method: measure the difference between old objective values and new ones
	vector <double> oldobj(nobj, 0.0);
	for (int i = 0; i < nobj; i++)
		oldobj[i] = population[n].indiv.y_obj[i];
	population[n].indiv.obj_eval();

	double Tolerror = 1.0e-4;
	for (int i = 0; i < nobj; i++)
	{
		if (fabs(population[n].indiv.y_obj[i] - oldobj[i])>Tolerror)
			return true;
	}
	return false;
}

// change reaction
void CSGEA::react_change()
{
	int size = population.size();
	// comput centroid of POS
	vector<double>  C(nvar, 0);
	C1.assign(nvar, 0);
	for (int n = 0; n < nvar; n++)
	{
		for (int i = 0; i < Archive.size(); i++)
			C1[n] += Archive[i].x_var[n];
		C1[n] /= Archive.size();
	}


	// requires a new start, so clear some old information
	Archive.clear();
	offspring.clear();

	// use maximal distance method to save 50% outdated population with good front diversity
	vector <int> accept, v2;

	int id1, id2, tmp;
	for (int i = 0; i < size; i++)
		v2.push_back(i);
	double min, max;

	//  find the extreme point on the y_obj[0] objective
	min = 1.0e+30, max = -1.0e+30;
	for (int i = 0; i<size - accept.size(); i++)
	{
		if (min>population[i].indiv.y_obj[0])
		{
			min = population[i].indiv.y_obj[0];
			id1 = i;
		}
		if (max < population[i].indiv.y_obj[0])
		{
			max = population[i].indiv.y_obj[0]; 
			id2 = i;
		}
	}
	tmp = v2[id1];
	accept.push_back(v2[id1]);
	v2[id1] = v2[size - accept.size()];
	v2[size - accept.size()] = tmp;

	if (id1 != id2)
	{
		tmp = v2[id2];
		accept.push_back(v2[id2]);
		v2[id2] = v2[size - accept.size()];
		v2[size - accept.size()] = tmp;
	}


	double *dist = new double[v2.size() - accept.size()];

	for (int i = 0; i < v2.size() - accept.size(); i++)
	{
		dist[i] = 1.0e+30;
		for (int j = 0; j<accept.size(); j++)
		{
			min = dist_vector(population[v2[i]].indiv.y_obj, population[accept[j]].indiv.y_obj);
			if (dist[i]>min) dist[i] = min;
		}
	}

	// preserve half of population using the-farthest-the-first method 
	while (accept.size() < size / 2)
	{
		max = -1.0e+30;
		for (int j = 0; j < v2.size() - accept.size(); j++)
		{
			if (max < dist[j])
			{
				max = dist[j];
				id2 = j;
			}
		}
		tmp = v2[id2];
		accept.push_back(v2[id2]);
		v2[id2] = v2[v2.size() - accept.size()];
		v2[v2.size() - accept.size()] = tmp;
		dist[id2] = dist[v2.size() - accept.size()];
		for (int j = 0; j<v2.size() - accept.size(); j++)
		{
			min = dist_vector(population[v2[j]].indiv.y_obj, population[tmp].indiv.y_obj);
			if (dist[j]>min) dist[j] = min;
		}

	}
	delete[] dist;

	vector<int> reject, predict, mark(size, 1);
	for (int i = 0; i < accept.size(); i++)
	{
		mark[accept[i]] = 0;
		population[accept[i]].indiv.obj_eval();
		fill_union(population[accept[i]].indiv);
	}


	// introducing #eta random or mutated solutions 
	for (int i = 0; i < size; i++)
	{
		if (mark[i])
		{
			reject.push_back(i);
			if (rnd_uni(&rnd_uni_init) < 2 * eta) // 2*eta;
			{
				population[i].indiv.rnd_init(); // random reinitialization
				//realmutation(population[i].indiv, 2.0 / nvar, 4); // alternative way is to mutate archive members.
				population[i].indiv.obj_eval();
				fill_union(population[i].indiv);
			}
			else
				predict.push_back(i); // obtain indices of members to be predicted
		}
	}

	// calculate moving stepsize
	double stepsize = 1.0;
	if (C0.size()) stepsize = dist_vector(C0, C1);

	vector <int> vnondom;
	vector <double> C2(nvar, 0);  // compute the global position of the population
	vnondom = rank_pop(offspring);

	// copy nondominated solutions to archive
	for (int i = 0; i < vnondom.size(); i++)
		Archive.push_back(offspring[vnondom[i]]);

	// calculate POS centroid for nondominated members and all population members
	for (int n = 0; n < nvar; n++)
	{
		for (int i = 0; i < vnondom.size(); i++)
			C[n] += offspring[vnondom[i]].x_var[n];
		C[n] /= vnondom.size();

		for (int i = 0; i < offspring.size(); i++)
			C2[n] += offspring[i].x_var[n];
		C2[n] /= offspring.size();
	}

	// re-initialize members by guided relocation formula
	for (int i = 0; i < predict.size(); i++)
	{
		if (C0.size())
			for (int n = 0; n < nvar; n++)
			{
				double normd = dist_vector(C, C2);
				normd = normd>0 ? normd : 1.0e-6;
				population[predict[i]].indiv.x_var[n] += stepsize*(C[n] - C2[n]) / normd + rnd_gaussian(&rnd_uni_init, 0, stepsize / sqrt(4 * nvar));
				if (population[predict[i]].indiv.x_var[n]<lowBound[n]) population[predict[i]].indiv.x_var[n] = lowBound[n];
				if (population[predict[i]].indiv.x_var[n]>uppBound[n]) population[predict[i]].indiv.x_var[n] = uppBound[n];
			}
		else
		{
			// random reinitialization for the first several generations (previous centroids are not available)
			population[predict[i]].indiv.rnd_init(); // random reinitialization
			//realmutation(population[predict[i]].indiv, 2.0 / nvar, 4); // mutation
		}
		population[predict[i]].indiv.obj_eval();
		fill_union(population[predict[i]].indiv);
		update_archive(population[predict[i]].indiv);
	}
	C0 = C1; // update past information
}

// print nondominated set
void CSGEA::save_front(char saveFilename[1024])
{
	std::fstream fout;
	fout.open(saveFilename, std::ios::out);
	for (int n = 0; n < Archive.size(); n++)
	{
		for (int k = 0; k < nobj; k++)
			fout << Archive[n].y_obj[k] << "  ";
		fout << "\n";
	}
	fout.close();
}

void CSGEA::save_ps(char saveFilename[1024])
{
	std::fstream fout;
	fout.open(saveFilename, std::ios::out);
	for (int n = 0; n < popsize; n++)
	{
		for (int k = 0; k < nvar; k++)
			fout << population[n].indiv.x_var[k] << "  ";
		fout << "\n";
	}
	fout.close();
}


void CSGEA::calc_distance()
{
	distance = 0;
	for (int i = 0; i < ps.size(); i++)
	{
		double min_d = 1.0e+10;
		for (int j = 0; j < population.size(); j++)
		{
			double d = dist_vector(ps[i].y_obj, population[j].indiv.y_obj);
			if (d < min_d)  min_d = d;
		}
		distance += min_d;
	}
	distance /= ps.size();
}

#endif