function MOEADTPN(Global)
% <algorithm> <H-N>
% An Improved Multiobjective Optimization Evolutionary Algorithm Based on 
% Decomposition for Complex Pareto Fronts
% delta --- 0.9 --- The probability of choosing parents locally
% nr    ---   2 --- Maximum number of solutions replaced by each offspring
% gr    --- 0.7 --- Percentage of computational resources allocated for the first stage
% nml   --- true --- the PF is a normalized one or with known nadir point
% operator      --- DE

%--------------------------------------------------------------------------
% Copyright (c) 2016-2017 BIMK Group. You are free to use the PlatEMO for
% research purposes. All publications which use this platform or any code
% in the platform should acknowledge the use of "PlatEMO" and reference "Ye
% Tian, Ran Cheng, Xingyi Zhang, and Yaochu Jin, PlatEMO: A MATLAB Platform
% for Evolutionary Multi-Objective Optimization [Educational Forum], IEEE
% Computational Intelligence Magazine, 2017, 12(4): 73-87".
%--------------------------------------------------------------------------

    %% Parameter setting
    [delta,nr, gr, nml] = Global.ParameterSet(0.9,2, 0.7,true);
    Tg=ceil(Global.maxgen*gr);
    optSense=1; % the smaller fitness the better
    %% Generate the weight vectors
    [W,Global.N] = UniformPoint(Global.N,Global.M);
    T = ceil(Global.N/10);

    %% Detect the neighbours of each solution
    B = pdist2(W,W);
    [~,B] = sort(B,2);
    B = B(:,1:T);

    %% Generate random population
    Population = Global.Initialization();
    Z = min(Population.objs,[],1);
    
    sigma=(Global.M-1)/Global.N; % or sigma=0.005 (sigma:niche sharing radius)
    arch=cell(Global.M,1); 
    %% Optimization
    while Global.NotTermination(Population)
        % find intercept
        if nml
            Intercept=ones(1,Global.M);
        else
            [Intercept,arch]=find_Intecept(Population,arch, Z);
        end
        
        % For each solution
        for i = 1 : Global.N
            % Choose the parents by niche-guided selection
            if (niche_count(i,Population, B, sigma, 1)>0.5*T && rand < 0.5)
                C=setdiff([1:Global.N], B(i,:));
                P=C(randperm(length(C)));
            else
                if rand < delta
                    P = B(i,randperm(size(B,2)));
                else
                    P = randperm(Global.N);
                end
            end

            % Generate an offspring
            Offspring = Global.Variation(Population([i,P(1:2)]),1,@DE);

            % Update the ideal point
            Z = min(Z,Offspring.obj);
            
            % Update the solutions in P by Inverted Tchebycheff approach
            if (optSense==1)
                g_old = max((Population(P).objs - repmat(Z, length(P),1))./repmat(Intercept-Z, length(P),1)./W(P,:),[],2);
                g_new = max(repmat((Offspring.obj-Z)./(Intercept-Z),length(P),1)./W(P,:),[],2);
            else
                g_old = min((repmat(Intercept,length(P),1)-Population(P).objs)./repmat(Intercept-Z, length(P),1)./W(P,:),[],2);
                g_new = min(repmat((Intercept-Offspring.obj)./(Intercept-Z),length(P),1)./W(P,:),[],2);
            end
            Population(P(find((g_old-g_new)*optSense>=0,nr))) = Offspring;
        end
        
        % Consider the second-stage search
        if Global.gen==Tg
            Archive=Population;
            boundary=prod(W,2)<0.5*(1/Global.M).^Global.M;
            
            % Solve the reverse problem
            if mean(niche_count(find(boundary),Population, B, sigma, 0))>= ...
                    mean(niche_count(find(~boundary),Population, B, sigma, 0))
                % reinitialize weight vector set
                W=(1-W)/(Global.M-1)./repmat(sum((1-W)/(Global.M-1),2), 1, Global.M);
                
                % reinitialize neighbourhood
                B = pdist2(W,W);
                [~,B] = sort(B,2);
                B = B(:,1:T);
                
                % find a matcher for each new weight vector
                Population=PopMatch(Population, Intercept, Z, W);
                optSense=-1;
            end
        end
        if (Global.evaluated>=Global.evaluation)
            Population=[Population,Archive];
            [~,IA]=unique([Population.objs, Population.cons],'rows');
            Population=Population(IA);
            Population(NDSort(Population.objs, Population.cons, 1)~=1)=[];
        end
    end
end

function niche=niche_count(ind, pop, B, sigma, flag)
for i=1:length(ind)
    id=ind(i);
    dist=pdist2(pop(id).objs,pop(B(id,:)).objs);
    tmp=dist<flag*sigma;
    dist(tmp)=1-(dist(tmp)/sigma).^2;
    niche(i)=sum(dist);
end
end

function pop=PopMatch(pop, zmax, zmin, W)
merged=pop;
[~,IA]=unique([merged.objs, merged.cons],'rows');
merged=merged(IA);
mergedObj=merged.objs;

mergedObj=(mergedObj-repmat(zmin, length(merged),1))./repmat(zmax-zmin, length(merged),1);

[N,M]=size(W);
record=true(N,1);
dist2 = pdist2(W-1.0/M,mergedObj);
k=0;
while (k<N)
    [dist,id]=min(dist2,[],1);
    if isinf(dist)
        pop(record)=merged(randperm(sum(record)));
        break;
    end
    for i=1:N
        if (record(i))
            idi=find(id==i);
            [~,s]=min(dist(idi));
            if ~isempty(s)
                pop(i)=merged(idi(s));
                record(i)=false;
                dist2(:,idi(s))=inf;
                dist2(i,:)=inf;
                k=k+1;
            end
        end
        if (k==N)
            break;
        end
    end 
end
end

function [intercept,arch]=find_Intecept(pop,arch, Zmin)

merged=[pop, [arch{:}]];
[~,IA]=unique([merged.objs, merged.cons],'rows');
merged=merged(IA);
popObj=merged.objs-repmat(Zmin,length(merged),1);
[N,M]  = size(popObj);
% Detect the extreme points and potentially extreme points
w = 1e-5+eye(M);
for i = 1 : M
    [~,I]=max(popObj./repmat(w(i,:),N,1),[],2);
    Potential=find(I==i);
    arch{i}=[arch{i},merged(Potential)];
    [~,IA]=unique([arch{i}.objs, arch{i}.cons],'rows');
    arch{i}=arch{i}(IA);
    arch{i}(NDSort(arch{i}.objs, arch{i}.cons, 1)~=1)=[];
    Y=popObj(Potential,i);
    if (length(Potential)>3)
        intercept(i)=min(Y);
    else
        intercept(i)=max(popObj(:,i),[],1);
    end

    if ~isempty(arch{i})
        archObj=arch{i}.objs-repmat(Zmin,length(arch{i}),1);
        intercept(i)=min(intercept(i), max(archObj(:,i)));
    end
end
intercept=intercept+Zmin;
end

