#
# $Id: PrintsHandle.pm,v 1.3 2002/01/07 16:24:54 phillord Exp $
#

package Prints::PrintsHandle;

use strict;
use Prints::Model::Sql::SqlPrintsHandle;
use Exporter;
use vars qw{@ISA};

my @ISA = qw{Exporter};


=head1

  This is the going to be an abstract factory, and a facade through which the factory
  works. The idea here is that we may want multiple implementations providing the same
  functionality. Its through this class that they should all operate.

  I'm not quite sure how this is all going to work yet. My current idea is that I should
  have a number of methods, which will define the inteface which all of the implementations
  should provide. And then additionally I'll provide a method which gives access to further
  information about implementation specific queries.

  I might also want to provide a couple of sets of standard interfaces so this is not
  all left down to the implementations. This might be the case for instance for
  writable implementations which is an obvious thing to provide.


=head2 Create Methods

  A whole bunch of create methods. These will check first whether the
  specific Prints::Model::$package::$packageBlah class exists. If it
  doesn't then it will use the basic Prints::Model::Blah class. These
  class will all be passed the "this" instance, which they are free to
  use if they need. Creators should be methods that take data from
  "outside" as it where, such as for instance a Swissprot accession,
  or a fingerprint accession number. They should take only a single
  parameter as well.

=head2 Search Methods

  Not sure how many of these I am going to need, but for the moment it makes
  sense just to provide a few trivial ones like "get_fingerprint_by_name",
  and so on. These will be anything that take more than a single parameter. 

=head2 Update Methods

  If I choose to have these then something this is for updating the underlying
  data source.

=head2 Optional Method examples

  Things like direct access to underlying data source. So for instance the SQL
  interface might allow direct SQL queries, returning string data.

=head3 new

  Instantiate a new PrintsHandles. Takes as hash parameter.
  Required hash parameters are...
  package :- the underlying package which will provide the implementation
  Option hash parameters are
  None:- All other parameters are passed directly to the underlying implementation,
  which can do as it pleases with them. 

=cut
sub new{
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $params = shift;

  # Start a new factory, which will provide the functionality for this class. 
  my $package = $params->{package};
  my $factory_name = "Prints::Model::" . $package . "::" . $package . "PrintsHandle";

  my $self =
    {
     package=>$package,
    };

  $params->{factory} = $self;
  $self->{factory} = $factory_name->new( $params );

  bless $self, $class;
  return $self;
}

sub _valid_creators{
  # these are all methods that will create new objects. The method
  # will take the as instance of this, and a hash containing all of
  # the other stuff, that it needs. All subclasses are required to
  # implement these. Perhaps I will write some code to ensure that
  # they do, at least when debug is switched on.
  return
     qw( create_fingerprint create_sequence );
}

sub _search_methods{
  # These will be all of the valid search methods that all
  # factories are required to implement.
  return
    qw( get_fingerprint_by_name );
}

sub is_updateable{
  my $self = shift;
  return $$self{factory}->is_updateable;
}

sub _update_methods{
  # Return all the updates that factories which return
  # true from the is_updateable method are required to implement.
  return
    qw( commit_object );
}

sub _optional_methods{
  my $self = shift;
  # return any optional methods.
  return $$self{factory}->_optional_methods;
}

sub _create_model_object{
  ## Will call this from the autoload function. Check whether there
  ## is a valid create_fingerprint method in the subclass factory. If
  ## just return the stardard. I may not need this later. It depends whether
  ## any of the objects are useful.

  my $self = shift;
  my $creator_name = shift;
  my $params = shift;

  return $$self{factory}->$creator_name( $params );
}

sub _delegate_search_method{
  ## Will call this from autoload function. Again will carry
  ## a link to this, and a reference
}

sub _delegate_update_method{
  ## guess what.
}

sub _delegate_optional_method{
  my $self = shift;
  my $method = shift;

  return $$self{factory}->$method( @_ );
}

sub _contains{
  my $self = shift;
  my $param = shift;
  my $method = shift;

  return scalar(grep{$_ eq $param} $self->$method() );
}

sub AUTOLOAD{
  my $self = shift;

  no strict;
  my $name = $AUTOLOAD;
  use strict;

  $name =~ s/.*://;

  if($name eq "DESTROY"){
    # we want to stop at this point all we get recursive loop
    # as we close down.
    return;
  }

  if($self->_contains( $name, "_valid_creators" )){
    return $self->_create_model_object( $name, \@_ );
  }

  elsif($self->_contains( $name, "_search_methods" )){
    return $self->_delegate_search_method( $name, \@_ );
  }

  elsif($self->is_updateable() && $self->_contains( $name, "_update_methods" )){
    return $self->_delegate_update_method( $name, \@_ );
  }

  elsif($self->_contains( $name, "_optional_methods" )){
    return $self->_delegate_optional_method( $name, \@_ );
  }
}

1;


