/*
 *  Copyright 2002 Phillip Lord, Victoria University of Manchester
 *
 *  This file is part of myGrid.  Further information, and the
 *  latest version, can be found at http://www.mygrid.info
 *
 *  myGrid is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as
 *  published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  myGrid is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with myGrid; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package uk.ac.man.cs.img.xml.merge; // Generated package name

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * SortDocument
 *
 * Sort the document subsequently to doing the merge. 
 *
 * The problem with ant is that some parts of it are procedural, while
 * some parts of it are declarative. So, for instance, a set of
 * targets will perform exactly the same in what ever order they
 * occur. On the flip side, properties will not, at least not in
 * general, if one property refers to another. 
 *
 * Generally ant files fall into a certain format though. You set the
 * properties at the beginning, then patterns, and some other top level
 * stuff, followed by all the targets. 
 *
 * Antmerge messes this up, which causes some problems. It often makes
 * a difference what order you put the parental files in, and acts in
 * a counter intuitive way. 
 *
 * This class doesn't solve all of these problems, but it restores a
 * lot of the ordering difficulties, by sorting the final document,
 * into an order which hopefully will "do what I mean". 
 *
 * That's the idea anyway. 
 *
 * Created: Thu Oct  2 21:21:26 2003
 *
 * @author Phillip Lord
 * @version $Id: SortDocument.java,v 1.1 2003/11/24 16:50:04 lordp Exp $
 */
public class SortDocument
{
    private static final SortPredicate[] predicates = new SortPredicate[]{
        new NameAttrPredicate( "property", "file" ),
        new NamePredicate( "property" ),
        new MultiNamePredicate
        ( new String[]{
            "patternset", "fileset", "path" 
        }),
        new NamePredicate( "taskdef" ),
        new NamePredicate( "target" )
    };
    
    private List[] elementLists;
    
    public SortDocument()
    {
        elementLists = new List[ predicates.length ];
        for( int i = 0; i < elementLists.length; i++ ){
            elementLists[ i ] = new ArrayList();
        }
    }
    
    public Document sortDocument( Document document )
    {
        // firstly get the parent document 
        Element projectNode = document.getDocumentElement();
        
        NodeList list = projectNode.getChildNodes();
        
        int length = list.getLength();
        for( int i = 0; i < length; i++ ){
            Node item = list.item( i );
            if( item.getNodeType() ==  Node.ELEMENT_NODE ){
                sortNode( item, projectNode );
            }
        }

        refillNodes( projectNode );
        
        return document;
    }

    protected void sortNode( Node item, Node projectNode )
    {
        for( int i = 0; i < predicates.length; i++ ){
            if( predicates[ i ].accept( (Element)item ) ){
                elementLists[ i ].add( item );
                return;
            }
        }
    }
    
    protected void refillNodes( Node projectNode )
    {
        for( int i = 0; i < predicates.length; i++ ){
            for( Iterator iter = elementLists[ i ].iterator();
                 iter.hasNext(); ){
                // append the node. Appending also removes any node if
                // they already occur (which they will)
                projectNode.appendChild
                    ( (Node)iter.next() );
            }
        }
        
    }
    
    public static interface SortPredicate
    {
        public boolean accept( Element element );
    }
    
    public static class NameAttrPredicate implements SortPredicate
    {
        private String name, attr;
        
        public NameAttrPredicate( String name, String attr )
        {
            this.name = name;
            this.attr = attr;
        }
        
        public boolean accept( Element element )
        {
            return element.getTagName().equals( name )
                && element.getAttributeNode( attr ) != null;
        }
    }
    
    public static class MultiNamePredicate implements SortPredicate
    {
        private String[] names;
        public MultiNamePredicate( String[] names )
        {
            this.names = names;
        }
        
        public boolean accept( Element element )
        {
            for( int i = 0; i < names.length; i++ ){
                if( element.getTagName().equals( names[ i ] ) )
                    return true;
            }
            return false;
        }
    }
    
    public static class NamePredicate implements SortPredicate
    {
        private String name;
        public NamePredicate( String name )
        {
            this.name = name;
        }
        
        public boolean accept( Element element )
        {
            return element.getTagName().equals( name );
        }
    }
} // SortDocument

/*
 * ChangeLog
 * $Log: SortDocument.java,v $
 * Revision 1.1  2003/11/24 16:50:04  lordp
 * Initial Checkin
 *
 *
 */
