;;; learn-language.el --- Helps to learn bilingual phrases 

;;
;;

;; This file is not part of Emacs

;; Author: Phillip Lord<p.lord@russet.org.uk>
;; Maintainer: Phillip Lord<p.lord@russet.org.uk>

;; COPYRIGHT NOTICE
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:
;;
;; This package helps you to learn a new language, by displaying
;; phrases in two languages in a dedicated frame. By having it in
;; Emacs, you can do a few every hour or so, and then go back to
;; work. Do this all day long, for a few months, and it should make a
;; significant impact on your vocabulary. 

;; Of course this might all be total rubbish, because since I decided
;; to learn another language, I have spent all of my time writing this
;; code, rather than, well, learning another languauge.

;; TODO:- The display software is now much better than it was,
;; although the display-slow, and display-quick functions are more or
;; less identical. I have put in a generic hook now, so that extra
;; displays can be added to, and I've used this in -spanish.el to add
;; speech support from festival. 

;; I've added the function of hiding the English phrase, which is
;; quite nice, but it's not put in very well...

;; The data structures need re-writing. The current system is
;; rubbish, and does lots of mucking around with symbols, all of which
;; needs removing. Also want to put all of the phrases into an
;; external data file, so that its easier to write, rather than having
;; to write into lisp data structures.


(require 'easy-mmode)
(require 'dframe)

(defvar learn-language-current-language "spanish")
(defvar learn-language-current-level  0)
(defvar learn-language-current-from-to nil)

(defvar learn-language-word-list nil)

(defun learn-language-check-language-loaded()
  "Is the language support file loaded yet"
  (require (intern
            (concat "learn-language-" learn-language-current-language))))

(defun learn-language-create-word-list()
  (interactive)
  (learn-language-check-language-loaded)
  (setq learn-language-current-from-to
        (symbol-value
         (intern (concat "learn-language-" 
                         learn-language-current-language
                         "-from-to"))))
  (setq learn-language-word-list nil)
  (let ((i 0))
    (while (and
            ;; we have run out of variables
            (> i -1)
            (<= i learn-language-current-level))
      (let ((level-variable-value
             (symbol-value 
              (intern-soft 
               (concat "learn-language-" 
                       learn-language-current-language 
                       "-" (number-to-string i))))))
        (if (not level-variable-value)
            (setq i -1)
          (incf i)
          (setq learn-language-word-list
                (append learn-language-word-list
                        level-variable-value))))))) 

(defun learn-language-add-lang-file(file)
  "Add a file in language speak format. "
  (interactive "fLoad Language File: ")
  (let ((count 0))
    (with-temp-buffer
      (insert-file file)
      (goto-char (point-min))
      (while
          (re-search-forward 
           (concat "\\(.*\\)" language-speak-delimiter "\\(.*\\)")
           (point-max) t)
        (incf count)
        (setq learn-language-word-list
            (append learn-language-word-list
                    (list (list (match-string 1)
                                (match-string 2)))))))
    (learn-language-message
     (format "Loaded File %s with %s words\n"
             file count) 'display)))

(defvar learn-language-last-phrase nil)

(defun learn-language-random-phrase()
  (if (not learn-language-word-list)
      (learn-language-create-word-list))
  (setq learn-language-last-phrase
        (nth (random
              (length learn-language-word-list))
             learn-language-word-list)))

(defvar learn-language-original-frame-title
  (frame-parameter (selected-frame) 'name))

(defvar learn-language-message-hook nil
  "This is not a normal hook!")

(defun learn-language-message-frame (phrase type)
  (let ((buffer (get-buffer "*Learn Language*")))
    (if buffer
        (save-window-excursion
          (set-buffer buffer)
          (select-frame learn-language-frame)
          (if (and 
               (equal type
                      (nth 0 learn-language-current-from-to))
               (not learn-language-show-from-language))
              (insert "phrase hidden")
            (insert phrase))))))

(defvar learn-language-show-from-language t)

(add-hook 'learn-language-message-hook
          'learn-language-message-frame)

(defun learn-language-message (phrase type)
  (run-hook-with-args 
   'learn-language-message-hook
   phrase type))


(defvar learn-language-buffer nil)
(defvar learn-language-frame nil)
(defvar learn-language-cached-frame nil)

(defvar learn-language-frame-parameters '(
                                          (width . 80)
                                          (height . 20)
                                          (border-width . 0)
                                          (menu-bar-lines . 0)
                                          (tool-bar-lines . 0)
                                          (unsplittable . t)))
  
(defvar learn-language-frame-mode-keymap nil)
(if learn-language-frame-mode-keymap
    ()
  (setq learn-language-frame-mode-keymap (make-sparse-keymap))
  (suppress-keymap learn-language-frame-mode-keymap t)
  ;; this unnecessary progn makes it easier to reeval the keymap
  (progn
    (define-key learn-language-frame-mode-keymap
      "u" 'learn-language-increase-level)
    (define-key learn-language-frame-mode-keymap
      "c" 'learn-language-clear-words)
    (define-key learn-language-frame-mode-keymap
      "d" 'learn-language-decrease-level)
    (define-key learn-language-frame-mode-keymap
      "f" 'learn-language-add-lang-file)
    (define-key learn-language-frame-mode-keymap
      "r"  'learn-language-repeat-last)
    (define-key learn-language-frame-mode-keymap
      "h" 'learn-language-toggle-show-to)      
    (define-key learn-language-frame-mode-keymap
      " " 'learn-language-noop)      
    (define-key learn-language-frame-mode-keymap
      [down] (lambda()
               (interactive)
               (learn-language-display-quick
                (learn-language-random-phrase)
                learn-language-current-from-to)))
    (define-key learn-language-frame-mode-keymap
      [right] 'learn-language-right-slow-translate)
    (define-key learn-language-frame-mode-keymap
      [left] 'learn-language-left-slow-translate)
    (define-key learn-language-frame-mode-keymap
      [up] 'learn-language-clear)))

(defun learn-language-noop()
  (interactive))
  
(defun learn-language-clear-words()
  (interactive)
  (learn-language-message "Clearing Word List\n" 'display)
  (learn-language-clear-list))

(defun learn-language-clear()
  (interactive)
  (let ((buffer (get-buffer "*Learn Language*")))
    (save-excursion
      (set-buffer buffer)
      (erase-buffer))))
      
(defun learn-language-repeat-last()
  (interactive)
  (let((learn-language-show-from-language t))
    (learn-language-display-quick
     learn-language-last-phrase
     learn-language-current-from-to)))

(defun learn-language-toggle-show-to()
  (interactive)
  (if learn-language-show-from-language
      (progn
        (setq learn-language-show-from-language nil)
        (learn-language-message
         (format "Hiding %s language phrases\n"
                 (symbol-name
                  (nth 0 learn-language-current-from-to)))
         'display))
    (setq learn-language-show-from-language t)
    (learn-language-message
     (format "Showing %s language phrases\n"
             (symbol-name
              (nth 0 learn-language-current-from-to)))
     'display)))

(defun learn-language-right-slow-translate()
  (interactive)
  (learn-language-display-slow
   (learn-language-random-phrase)
   learn-language-current-from-to))

(defun learn-language-left-slow-translate()
  (interactive)
  (learn-language-display-slow
   (reverse 
    (learn-language-random-phrase))
   (reverse 
    learn-language-current-from-to)))

(defun learn-language-display-slow(phrase from-to)
  (interactive)
  (learn-language-message 
   (car phrase) 
   (car from-to))
  (learn-language-message " :- \n\t" 'display)
  (sit-for 10)
  (learn-language-message
   (car (cdr phrase))
   (car (cdr from-to)))
  (learn-language-message 
   (format " (%s) \n" 
           learn-language-current-language)
   'display))

(defun learn-language-display-quick(phrase from-to)
  (learn-language-message 
   (car phrase) 
   (car from-to))
  (learn-language-message " :- \n\t" 'display)
  (learn-language-message
   (car (cdr phrase))
   (car (cdr from-to)))
  (learn-language-message 
   (format " (%s) \n" 
           learn-language-current-language)
   'display))

(defalias 'learn-language 'learn-language-frame)

(defun learn-language-frame(&optional arg)
  (interactive "P")
  (if (not (buffer-live-p learn-language-buffer))
      (save-excursion
        (setq learn-language-buffer 
              (get-buffer-create "*Learn Language*"))
        (set-buffer learn-language-buffer)
        (learn-language-frame-mode)))
  (dframe-frame-mode arg
                     'learn-language-frame
                     'learn-language-cached-frame
                     'learn-language-buffer
                     "Learn Language"
                     #'learn-language-frame-mode
                     learn-language-frame-parameters))


(defun learn-language-frame-mode()
  "Mode for helping to learn a language"
  (interactive)
  (kill-all-local-variables)
  (use-local-map learn-language-frame-mode-keymap)
  (setq major-mode 'learn-language-frame-mode)
  (setq mode-name "learn language frame"))

(defun learn-language-increase-level()
  (interactive)
  (setq learn-language-word-list nil)
  (learn-language-message
   (format "Language Level now at %s \n"
           (incf learn-language-current-level))
   'display))

(defun learn-language-decrease-level()
  (interactive)
  (setq learn-language-word-list nil)
  (incf learn-language-current-level -1)
  (if (< learn-language-current-level 0)
      (setq learn-language-current-level 0))
  (learn-language-message
   (format "Language level now at %s \n" 
           learn-language-current-level)
   'display))

(defun learn-language-clear-list()
  (interactive)
  (setq learn-language-word-list nil))

(provide 'learn-language)