// File: YearRange.java

/**
 * This class represents a date that might not be precisely known.
 * The date is stored as an earliest and latest possible year.
 * 
 * @author Vegard Brox
 */
public class YearRange
{
   /**
    * A constant value used to indicate that the date was before
    * a specified year.
    */
   public static final String BEFORE   = "before";
   /**
    * A constant value used to indicate that the date was after
    * a specified year.
    */
   public static final String AFTER    = "after";

   // Default start and end of the year range if nothing more precise is known
   private static final int DEFAULT_START = 0;
   private static final int DEFAULT_END   = 9999;

   // Private variables for start and end of the year range
   private int myStart;
   private int myEnd;
   private boolean myChanged = false;

   /**
    * Constructor that initialises the start and the end of the year range
    * to the specified values. If the same value is used for both parameters,
    * this year range will represent a specific year.
    *
    * @param start The start of the year range
    * @param end The end of the year range
    */
   public YearRange( int start, int end )
   {
      myStart = start;
      myEnd = end;
   }
   
   /**
    * Constructor that initialises either the start or the end of the
    * year range to the specified value, and then uses the default value
    * for the other year.
    *
    * @param mode Either BEFORE to specify the end of the year range or
    *             BEFORE to specify the start of the year range
    * @param year The year to initialise either start or end of the 
    *             year range with
    */
   public YearRange( String mode, int year )
   {
      // If before mode - store the end
      if( mode != null && mode.equals( BEFORE ) )
      {
         myStart = DEFAULT_START;
         myEnd = year;
      }
      // If after mode - store the start
      else if( mode != null && mode.equals( AFTER ) )
      {
         myStart = year;
         myEnd = DEFAULT_END;
      }
      // If anything else given as parameter - use default values
      else
      {
         myStart = DEFAULT_START;
         myEnd = DEFAULT_END;
      }
   }
   
   /**
    * Parameterless constructor that initialises the year range with 
    * default values (0-9999).
    */
   public YearRange()
   {
      myStart = DEFAULT_START;
      myEnd = DEFAULT_END;
   }
   
   /**
    * Returns the start of the year range
    * @return The start of the year range
    */
   public int getStart()
   {
      return myStart;
   }
   
   /**
    * Return the end of the year range
    * @return The end of the year range
    */
   public int getEnd()
   {
      return myEnd;
   }
   
   /**
    * Set the start of the year range
    * @param value The start of the year range
    */
   public void setStart( int value )
   {
      myStart = value;
      myChanged = true;
   }
   
   /**
    * Set the end of the year range
    * @param value The end of the year range
    */
   public void setEnd( int value )
   {
      myEnd = value;
      myChanged = true;
   }
   
   /**
    * Returns whether this year range has been changed since
    * it was created.
    * @return true if this year range has been changed, false otherwise
    */
   public boolean isChanged()
   {
      return myChanged;
   }
   
   /**
    * Returns whether this year range is negative (i.e the end
    * is before the start)
    * @return true if this year range is negative, false otherwise
    */
   public boolean isNegative()
   {
      return ( myEnd < myStart );
   }
   
   /**
    * Make a string representation of this object
    * @return A string representation of this object
    */
   public String toString()
   {
      return "From " + myStart + " to " + myEnd;
   }
}
