// File: OptionsWindow.java

// Import
import java.awt.Button;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Label;
import java.awt.Panel;
import java.awt.SystemColor;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;


/**
 * This class represents a window with options for the GEDCOM Locater.
 * 
 * @author Vegard Brox
 */
public class OptionsWindow extends Frame
{
   // Private constants
   private static final String TITLE         = "GEDCOM Estimator Options";
   private static final String ACTION_OK     = "OK";
   private static final String ACTION_CANCEL = "CANCEL";

   // Private variables
   private TextField myMaxLivingAge;
   private TextField myMaxChristening;
   private TextField myMaxBurial;
   private TextField myMinMarriage;
   private TextField myMinChildBirth;
   private TextField myMaxChildBirth;
   private TextField myAboutValue;

   /**
    * Creates a new, initially invisible, options window.
    *
    * @param x Initial x location
    * @param y Initial y location
    */
   public OptionsWindow( int x, int y )
   {
      // Calling Frame constructor
      super( TITLE );
      
      // Creating event handler and properties object
      MyEventHandler eventHandler = new MyEventHandler();
      EstimationProperties properties = EstimationProperties.getInstance();
      
      // Creating panels
      Panel labelPanel = new Panel( new GridLayout( 7, 1 ) );
      Panel textPanel = new Panel( new GridLayout( 7, 1 ) );
      Panel yearPanel = new Panel( new GridLayout( 7, 1 ) );
      Panel buttonPanel = new Panel();

      // Creating header labels
      labelPanel.add( new Label( "Maximum living age:" ) );
      labelPanel.add( new Label( "Maximum christening age:" ) );
      labelPanel.add( new Label( "Maximum  time between death and burial:" ) );
      labelPanel.add( new Label( "Minimum marriage age:" ) );
      labelPanel.add( new Label( "Minimum age to have child:" ) );
      labelPanel.add( new Label( "Maximum age for woman to give birth:" ) );
      labelPanel.add( new Label( "ABT means:" ) );

      // Retrieving stored options
      int maxLivingAge = properties.getProperty( "maxLivingAge" );
      int maxChristening = properties.getProperty( "maxChristening" );
      int maxBurial = properties.getProperty( "maxBurial" );
      int minMarriage = properties.getProperty( "minMarriage" );
      int minChildBirth = properties.getProperty( "minChildBirth" );
      int maxChildBirth = properties.getProperty( "maxChildBirth" );
      int aboutValue = properties.getProperty( "aboutValue" );

      // Creating text fields with the stored options
      myMaxLivingAge = new TextField( String.valueOf( maxLivingAge ) );
      myMaxChristening = new TextField( String.valueOf( maxChristening ) );
      myMaxBurial = new TextField( String.valueOf( maxBurial ) );
      myMinMarriage = new TextField( String.valueOf( minMarriage ) );
      myMinChildBirth = new TextField(String.valueOf( minChildBirth ) );
      myMaxChildBirth = new TextField( String.valueOf( maxChildBirth ) );
      myAboutValue = new TextField( String.valueOf( aboutValue ) );

      // Adding text fields to panel
      textPanel.add( myMaxLivingAge );
      textPanel.add( myMaxChristening );
      textPanel.add( myMaxBurial );
      textPanel.add( myMinMarriage );
      textPanel.add( myMinChildBirth );
      textPanel.add( myMaxChildBirth );
      textPanel.add( myAboutValue );

      // Creating trailer labels
      for( int i=0; i<7; i++ )
      {
         yearPanel.add( new Label( "year(s)" ) );
      }
      
      // Creating buttons
      Button okButton = new Button( "   OK   " );
      okButton.setActionCommand( ACTION_OK );
      okButton.addActionListener( eventHandler );
      Button cancelButton = new Button( " Cancel " );
      cancelButton.setActionCommand( ACTION_CANCEL );
      cancelButton.addActionListener( eventHandler );
      buttonPanel.add( okButton );
      buttonPanel.add( cancelButton );
      
      // Adding everything to frame
      add( "West", labelPanel );
      add( "Center", textPanel );
      add( "East", yearPanel );
      add( "South", buttonPanel );
      setBackground( SystemColor.menu );
      pack();

      // Set location
      setLocation( x, y );
   }

   /**
    * The event handler class for the options window
    */
   private class MyEventHandler implements ActionListener
   {
      /**
       * This method is called when the user presses OK or Cancel.
       * If the user pressed OK, the options are stored and the 
       * window closed, otherwise the window is closed without 
       * storing options.
       *
       * @param e The action event
       */
      public void actionPerformed( ActionEvent e )
      {
         // Get the command
         String command = e.getActionCommand();
         
         if( command.equals( ACTION_OK ) )
         {
            // If command is OK:
            // Get properties
            EstimationProperties properties = 
                  EstimationProperties.getInstance();
            
            try
            {
               // Set new properties
               properties.setProperty( "maxLivingAge", 
                     Integer.parseInt( myMaxLivingAge.getText() ) );
               properties.setProperty( "maxChristening", 
                     Integer.parseInt( myMaxChristening.getText() ) );
               properties.setProperty( "maxBurial", 
                     Integer.parseInt( myMaxBurial.getText() ) );
               properties.setProperty( "minMarriage", 
                     Integer.parseInt( myMinMarriage.getText() ) );
               properties.setProperty( "minChildBirth", 
                     Integer.parseInt( myMinChildBirth.getText() ) );
               properties.setProperty( "maxChildBirth", 
                     Integer.parseInt( myMaxChildBirth.getText() ) );
               properties.setProperty( "aboutValue", 
                     Integer.parseInt( myAboutValue.getText() ) );
               
               // Save and close window
               properties.saveProperties();
               OptionsWindow.this.dispose();
            }
            catch( NumberFormatException nfe )
            {
               // If the user gave illeagal values - give error message
               MessageDialog dialog = new MessageDialog( 
                     OptionsWindow.this, 
                     "GEDCOM Estimator Error",
                     "Illegal number value! " +
                     "The values can only be whole numbers.",
                     MessageDialog.ERROR );
               dialog.show();
            }
         }
         else
         {
            // If the command is Cancel - close window
            OptionsWindow.this.dispose();
         }
      }
   }
}
