// File: MessageDialog.java

// Import
import java.awt.Button;
import java.awt.Dialog;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Panel;
import java.awt.SystemColor;
import java.awt.TextArea;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;


/**
 * This class represents a message dialog for an error or info message.
 *
 * @author Vegard Brox
 */
public class MessageDialog extends Dialog
{
   // Public constants
   /**
    * Indicates that this is an info message box
    */
   public static final int INFO  = 1;
   /**
    * Indicates that this is an error message box
    */
   public static final int ERROR = 2;
   
   // Private constants
   private static final String BUTTON_OK  = "   OK   ";
   private static final int LINE_LENGTH   = 40;
   
   // Private variables
   private Button myOKButton;
   
   /**
    * Creates a new, initially invisible, message dialog.
    *
    * @param parent The parent frame of the dialog
    * @param title The title of the dialog
    * @param message The message to display
    * @param type The type of message, either INFO or ERROR
    */
   public MessageDialog( Frame parent, String title, String message, int type )
   {
      // Calling Dialog constructor
      super( parent, title, true );
      
      // Creating event handler
      MyEventHandler eventHandler = new MyEventHandler();
      
      // Format text and create text area
      message = formatText( message );
      TextArea textArea = new TextArea( message, 
                                        message.length() / LINE_LENGTH + 2, 
                                        35, 
                                        TextArea.SCROLLBARS_NONE );
      textArea.setBackground( SystemColor.menu );
      textArea.setEditable( false );

      // Create OK button
      myOKButton = new Button( BUTTON_OK );
      myOKButton.addActionListener( eventHandler );
      Font font = new Font( "SansSerif", Font.BOLD, 12 );
      myOKButton.setFont( font );
      Panel buttonPanel = new Panel();
      buttonPanel.add( myOKButton );

      // Add to dialog
      add( "Center", textArea );
      add( "South", buttonPanel );
      pack();
      
      // Set location
      setLocation( parent.getLocation().x - 10, parent.getLocation().y + 30 );
      
      // Add window listener
      addWindowListener( eventHandler );
   }
   
   /**
    * Inserts line breaks in a string to make it easier to display it.
    *
    * @param text The text to format
    * @return The formatted text
    */
   private String formatText( String text )
   {
      String formattedText = "";
      while( text.length() > LINE_LENGTH )
      {
         int index = text.lastIndexOf( " ", LINE_LENGTH );
         formattedText += text.substring( 0, index ) + "\n";
         text = text.substring( index + 1 );
      }
      formattedText += text;
      return formattedText;
   }
   
   /**
    * Event handler class for the message dialog
    */
   private class MyEventHandler extends WindowAdapter 
                                implements ActionListener
   {
      /**
       * Called only when the OK button is pressed. Closes the dialog.
       *
       * @param e The action event
       */
      public void actionPerformed( ActionEvent e )
      {
         MessageDialog.this.dispose();
      }
      
      public void windowOpened( WindowEvent e )
      {
         myOKButton.requestFocus();
      }
      
      public void windowActivated( WindowEvent e )
      {
         myOKButton.requestFocus();
      }
   }
}
