// File: MainWindow.java

// Import
import java.awt.BorderLayout;
import java.awt.Button;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Label;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.Panel;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.TextArea;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;


/**
 * This class represents the main window in the GEDCOM Locater application.
 * It can start the estimation procedure, as well as open new windows with
 * options, help etc.
 *
 * @author Vegard Brox
 */
public class MainWindow extends Frame
{
   // Private constants
   private static final String DEFAULT_TITLE       = "GEDCOM Estimator";

   private static final String GUI_LOAD_TITLE      = "Select input GEDCOM file:";
   private static final String GUI_SAVE_TITLE      = "Save the output GEDCOM file as:";
   private static final String GUI_BUTTON_START    = "Start estimation";
   private static final String GUI_BUTTON_OPTIONS  = "Set options";
   private static final String GUI_ERROR           = DEFAULT_TITLE + " Error";
   private static final String GUI_INFO            = DEFAULT_TITLE + " Info";

   private static final String ACTION_START        = "START";
   private static final String ACTION_OPTIONS      = "OPTIONS";
   private static final String ACTION_HELP         = "HELP";
   private static final String ACTION_ABOUT        = "ABOUT";
   private static final String ACTION_EXIT         = "EXIT";

   // Private variables
   private TextArea myStatusText;


   /**
    * Constructs a new, initially invisible, main window.
    */
   public MainWindow()
   {
      // Calling Frame constructor
      super( DEFAULT_TITLE );

      // Construct event handler
      MyEventHandler eventHandler = new MyEventHandler();
      
      // Make menues
      Menu fileMenu = new Menu( "File" );
      MenuItem openFile = new MenuItem( GUI_BUTTON_START );
      openFile.setActionCommand( ACTION_START );
      MenuItem exit = new MenuItem( "Exit" );
      exit.setActionCommand( ACTION_EXIT );
      fileMenu.add( openFile );
      fileMenu.add( exit );
      fileMenu.addActionListener( eventHandler );
      Menu optionsMenu = new Menu( "Options" );
      MenuItem changeOptions = new MenuItem( GUI_BUTTON_OPTIONS );
      changeOptions.setActionCommand( ACTION_OPTIONS );
      optionsMenu.add( changeOptions );
      optionsMenu.addActionListener( eventHandler );
      Menu helpMenu = new Menu( "Help" );
      MenuItem helpContents = new MenuItem( "Help contents" );
      helpContents.setActionCommand( ACTION_HELP );
      MenuItem about = new MenuItem( "About" );
      about.setActionCommand( ACTION_ABOUT );
      helpMenu.add( helpContents );
      helpMenu.add( about );
      helpMenu.addActionListener( eventHandler );
      
      // Combine menues to menubar
      MenuBar menuBar = new MenuBar();
      menuBar.add( fileMenu );
      menuBar.add( optionsMenu );
      menuBar.setHelpMenu( helpMenu );
      setMenuBar( menuBar );
   
      // Construct buttons
      Button start = new Button( GUI_BUTTON_START );
      Button options = new Button( GUI_BUTTON_OPTIONS );

      // Add action commands and action listeners to buttons
      start.setActionCommand( ACTION_START );
      start.addActionListener( eventHandler );
      options.setActionCommand( ACTION_OPTIONS );
      options.addActionListener( eventHandler );
      
      // Make font for buttons and assign it
      Font startFont = new Font( "SansSerif", Font.BOLD, 20 );
      Font buttonFont = new Font( "SansSerif", Font.PLAIN, 16 );
      start.setFont( startFont );
      options.setFont( buttonFont );

      // Construct text area
      myStatusText = new TextArea( 5, 30 );
      myStatusText.setEditable( false );
      myStatusText.setBackground( SystemColor.menu );
      
      // Combine to panels
      Panel startPanel = new Panel();
      startPanel.add( start );
      Panel buttonsPanel = new Panel();
      buttonsPanel.add( options );
      Panel southPanel = new Panel( new BorderLayout() );
      southPanel.add( "Center", buttonsPanel );
      southPanel.add( "South", myStatusText );

      // Add everything to frame
      add( "North", new Label( " " ) );
      add( "Center", startPanel );
      add( "South", southPanel );
      setBackground( SystemColor.menu );
      pack();

      // Calulate and set location for frame
      int screenX = getToolkit().getScreenSize().width;
      int screenY = getToolkit().getScreenSize().height;
      int frameX = getSize().width;
      int frameY = getSize().height;
      setLocation( screenX/2 - frameX/2, screenY/2 - frameY/2 );
   }
   
   /**
    * This procedure corresponds to the main program of the command-line
    * version of the program. It asks the user for an input file, reads
    * the file, performs the estimation, asks the user for an output file,
    * and writes it.
    */
   private void performEstimation()
   {
      // Ask user for input file and create file object
      FileDialog selectInput = new FileDialog( this, GUI_LOAD_TITLE, 
                                               FileDialog.LOAD );
      selectInput.show();
      if( selectInput.getFile() == null )
      {
         myStatusText.append( "Estimation process cancelled\n" );
         return;
      }
      File inFile = new File( selectInput.getDirectory(), 
                              selectInput.getFile() );

      // Create hashtables
      Hashtable people = new Hashtable();
      Hashtable families = new Hashtable();
      
      try
      {
         // Try to read the file
         myStatusText.append( "Reading input file...\n" );
         GEDCOMInput.input( inFile, people, families );
      }
      catch( FileNotFoundException e )
      {
         // Give error message if the GEDCOM file was not found
         MessageDialog errorDialog = new MessageDialog( this, GUI_ERROR, 
                           "ERROR! The specified GEDCOM file was not found",
                           MessageDialog.ERROR );
         errorDialog.show();
         return;
      }
      
      // Create database
      myStatusText.append( "Creating database...\n" );
      GenealogyDB database = new GenealogyDB( people, families );

      // Perform estimation
      myStatusText.append( "Estimating dates...\n" );
      Estimator estimator = new Estimator( database );
      int iterations = estimator.estimate();
      myStatusText.append( "Estimation complete after " + 
                          iterations + " iterations\n" );

      // Get subsets with errors and prepare for writing error files
      Enumeration errorDBs = estimator.getErrorDBs();
      int errorNo = 0;
      String fileName = inFile.getName();
      int dotIndex = fileName.lastIndexOf( "." );
      if( dotIndex < 0 )
      {
         dotIndex = fileName.length();
      }
      String baseFileName = fileName.substring( 0, dotIndex ) + ".err";

      // Loop through subsets and write each one to file
      while( errorDBs.hasMoreElements() )
      {
         GenealogyDB errorDB = (GenealogyDB)errorDBs.nextElement();
         String newFileName = baseFileName + (++errorNo) + ".ged";
         File errFile = new File( selectInput.getDirectory(), newFileName );
         try
         {
            GEDCOMOutput errorOutput = new GEDCOMOutput( inFile, errFile, 
                                                         errorDB, true );
            errorOutput.output();
         }
         catch( IOException e )
         {
            MessageDialog errorDialog = new MessageDialog( this, GUI_ERROR,
                           "ERROR! Could not write subset file. Details:" +
                           e.getMessage(),
                           MessageDialog.ERROR );
            errorDialog.show();
         }
      }
      if( errorNo > 0 )
      {
         MessageDialog dialog = new MessageDialog( this, GUI_INFO,
                        errorNo + " negative year range(s) where " +
                        "encountered during estimation. " +
                        "Subset(s) of GEDCOM code containing one or " +
                        "more inconsistencies have been saved in file(s) "+
                        "with name(s) " + baseFileName + "1.ged etc.",
                        MessageDialog.INFO );
         dialog.show();
      }

      // Ask user for output file and create file object
      FileDialog selectOutput = new FileDialog( this, GUI_SAVE_TITLE, 
                                                FileDialog.SAVE );
      selectOutput.show();
      if( selectOutput.getFile() == null )
      {
         myStatusText.append( "Estimation process cancelled\n" );
         return;
      }
      File outFile = new File( selectOutput.getDirectory(), 
                               selectOutput.getFile() );

      try
      {
         // Write output file
         myStatusText.append( "Writing to file " + outFile.getName() + "...\n" );
         GEDCOMOutput output = new GEDCOMOutput( inFile, outFile, database );
         output.output();
      }
      catch( IOException e )
      {
         MessageDialog errorDialog = new MessageDialog( this, GUI_ERROR,
                           "ERROR! Could not write the output file. Details:" +
                           e.getMessage(),
                           MessageDialog.ERROR );
         errorDialog.show();
         return;
      }
      myStatusText.append( "Finshed writing to file\n" );
      
   }
   
   /**
    * Event handler class for the main window.
    */
   private class MyEventHandler implements ActionListener
   {
      /**
       * This method is called when the user has performed some action.
       * The parameter is used to determine what action is what, and
       * then appropriate action is taken.
       *
       * @param e The action event for the action the user has performed
       */
      public void actionPerformed( ActionEvent e )
      {
         // Get the type of action
         String command = e.getActionCommand();
         
         // If action is start estimation - call method for estimation
         if( command.equals( ACTION_START ) )
         {
            performEstimation();
         }
         // If action is set options - open options window
         else if( command.equals( ACTION_OPTIONS ) )
         {
            Point location = MainWindow.this.getLocation();
            OptionsWindow options = new OptionsWindow( location.x - 10, 
                                                       location.y + 30 );
            options.show();
         }
         // If action is help - only message box so far
         else if( command.equals( ACTION_HELP ) )
         {
            MessageDialog dialog = new MessageDialog( 
                  MainWindow.this,
                  "GEDCOM Estimator Information",
                  "Please open the file \"help.html\" to start the help system!",
                  MessageDialog.INFO );
            dialog.show();
         }
         // If action is about - open about box
         else if( command.equals( ACTION_ABOUT ) )
         {
            AboutBox about = new AboutBox( MainWindow.this );
            about.show();
         }
         // If action is exit - close window
         else if( command.equals( ACTION_EXIT ) )
         {
            MainWindow.this.dispose();
         }
         // If action is not recognised - print error message
         else
         {
            System.err.println( "Unrecognised command" );
         }
      }
   }
}
