// File: MainProgram.java

// Import
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * Main program for the command line version of the program
 *
 * @author Vegard Brox
 */
public class MainProgram
{
   private static final String DEBUG   = "DEBUG";

   // Not supposed to create instances of this class
   private MainProgram()
   {}

   /**
    * The main program.
    *
    * @param args The command-line arguments. Filename of the GEDCOM file is
    *             expected as the first argument
    */
   public static void main( String args[] )
   {
      // If no GEDCOM file specified - print error message and exit
      if( args.length < 2 )
      {
         System.out.println( "Usage: java MainProgram <inputfile> <outputfile>" );
         System.exit(0);
      }
      
      // Set debug flag
      boolean debug = false;
      if( args.length > 2 && args[2] != null && args[2].equals( DEBUG ) )
      {
         debug = true;
      }
   
      // Create file objects
      File inFile = new File( System.getProperty( "user.dir" ), args[0] );
      File outFile = new File( System.getProperty( "user.dir" ), args[1] );
      
      // Create hashtables
      Hashtable people = new Hashtable();
      Hashtable families = new Hashtable();

      try
      {
         // Try to read the file
         System.out.println( "Reading " + inFile.getName() + "..." );
         GEDCOMInput.input( inFile, people, families );
      }
      catch( FileNotFoundException e )
      {
         // Print error message if the GEDCOM file was not found
         System.out.println( "ERROR: The specified GEDCOM file was not found" );
         System.exit(0);
      }
      
      // Create database
      System.out.println( "Creating database..." );
      GenealogyDB database = new GenealogyDB( people, families );

      // If debug mode - print database
      if( debug )
      {
         DebugOutput debugOutput = new DebugOutput( database );
         debugOutput.printPeople();
         debugOutput.printFamilies();
      }

      // Perform estimation
      System.out.println( "Estimating dates..." );
      Estimator estimator = new Estimator( database );
      int iterations = estimator.estimate();
      System.out.println( "Estimation complete after " + 
                             iterations + " iterations" );
      if( debug )
      {
         Hashtable log = estimator.getLog();
         DebugOutput.printLog( log );
      }
         
      // Get subsets with errors and prepare for writing error files
      Enumeration errorDBs = estimator.getErrorDBs();
      int errorNo = 0;
      String fileName = inFile.getName();
      int dotIndex = fileName.lastIndexOf( "." );
      if( dotIndex < 0 )
      {
         dotIndex = fileName.length();
      }
      String baseFileName = fileName.substring( 0, dotIndex ) + ".err";

      // Loop through subsets and write each one to file
      while( errorDBs.hasMoreElements() )
      {
         GenealogyDB errorDB = (GenealogyDB)errorDBs.nextElement();
         String newFileName = baseFileName + (++errorNo) + ".ged";
         File errFile = new File( System.getProperty( "user.dir" ), 
                                  newFileName );
         try
         {
            GEDCOMOutput errorOutput = new GEDCOMOutput( inFile, errFile, 
                                                         errorDB, true );
            errorOutput.output();
         }
         catch( IOException e )
         {
            System.err.println( "ERROR! Could not writer file. Details:" );
            System.err.println( e.getMessage() );
         }
      }
      if( errorNo > 0 )
      {
         System.out.println();
         System.out.println( errorNo + " negative year range(s) where " +
                             "encountered during estimation." );
         System.out.println( "Subset(s) of GEDCOM code containing one or " +
                             "more inconsistencies have been saved in file(s) "+
                             "with name(s) " + baseFileName + "1.ged etc." );
         System.out.println();
      }
      
      try
      {
         // Write output file
         System.out.println( "Writing to file " + outFile.getName() + "..." );
         GEDCOMOutput output = new GEDCOMOutput( inFile, outFile, database );
         output.output();
      }
      catch( IOException e )
      {
         System.err.println( "ERROR! Could not write the output file. Details:" );
         System.err.println( e.getMessage() );
         System.exit(0);
      }
      System.out.println( "Program finished." );
   }
}
