// File: GenealogyDB.java

// Import
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;


/**
 * This class holds an internal representation of a GEDCOM file.
 * The data are held in two hashtables, one for people and one for families.
 * These hashtables can be accessed directly, but there are also utility
 * functions available to find e.g. the parents of a given individual.
 *
 * @author Vegard Brox
 */
public class GenealogyDB
{
   // Private variables
   private Hashtable myPeople;
   private Hashtable myFamilies;

   /**
    * Constructs a new genealogy database initialised with the 
    * given hashtables.
    *
    * @param people A hashtable initialised with Person objects
    * @param families A hastable initialised with Family objects
    */
   public GenealogyDB( Hashtable people, Hashtable families )
   {
      myPeople = people;
      myFamilies = families;
   }
   
   /**
    * Returns the hashtable with people.
    *
    * @return A hashtable with Person objects
    */
   public Hashtable getPeople()
   {
      return myPeople;
   }
   
   /**
    * Returns the hashtable with families.
    * 
    * @return A hashtable with Family objects
    */
   public Hashtable getFamilies()
   {
      return myFamilies;
   }
   
   /**
    * Finds the father of a given individual.
    *
    * @param personID ID of the person to find the father for
    * @return A Person object representing the father
    */
   public Person getFather( String personID )
   {
      Person person = (Person)myPeople.get( personID );
      if( person != null )
      {
         String familyID = person.getCFamily();
         if( familyID != null )
         {
            return getHusband( familyID );
         }
      }
      return null;
   }
   
   /**
    * Finds the mother of a given individual.
    * 
    * @param personID ID of the person to find the mother for
    * @return A Person object representing the mother
    */
   public Person getMother( String personID )
   {
      Person person = (Person)myPeople.get( personID );
      if( person != null )
      {
         String familyID = person.getCFamily();
         if( familyID != null )
         {
            return getWife( familyID );
         }
      }
      return null;
   }
   
   /**
    * Finds the families in which the given individual is a spouse.
    * 
    * @param personID ID of the person to find families for
    * @return An enumeration of Family objects representing the families
    */
   public Enumeration getSFamilies( String personID )
   {
      Vector families = new Vector();
      Person person = (Person)myPeople.get( personID );
      if( person != null )
      {
         Enumeration familyIDs = person.getSFamilies();
         while( familyIDs.hasMoreElements() )
         {
            Family family = (Family)myFamilies.get( familyIDs.nextElement() );
            if( family != null )
            {
               families.addElement( family );
            }
         }
      }
      return families.elements();
   }
   
   /**
    * Finds the children in a given family.
    * 
    * @param familyID ID of the family to find the children for
    * @return An enumeration of Person objects representing the children
    */
   public Enumeration getChildren( String familyID )
   {
      Family family = (Family)myFamilies.get( familyID );
      Vector children = new Vector();
      if( family != null )
      {
         Enumeration childrenIDs = family.getChildren();
         while( childrenIDs.hasMoreElements() )
         {
            Person child = (Person)myPeople.get( childrenIDs.nextElement() );
            if( child != null )
            {
               children.addElement( child );
            }
         }
      }
      return children.elements();
   }

   /**
    * Finds the husband in a given family.
    * 
    * @param familyID ID of the family to find husband for
    * @return A Person object representing the husband
    */
   public Person getHusband( String familyID )
   {
      Family family = (Family)myFamilies.get( familyID );
      if( family != null )
      {
         String husbandID = family.getHusband();
         if( husbandID != null )
         {
            return (Person)myPeople.get( husbandID );
         }
      }
      return null;
   }
   
   /**
    * Finds the wife in a given family.
    * 
    * @param familyID ID of the family to find the wife for
    * @return A Person object representing the wife
    */
   public Person getWife( String familyID )
   {
      Family family = (Family)myFamilies.get( familyID );
      if( family != null )
      {
         String wifeID = family.getWife();
         if( wifeID != null )
         {
            return (Person)myPeople.get( wifeID );
         }
      }
      return null;
   }
}

