// File: GEDCOMInput.java

// Import
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/** 
 * This class writes a GEDCOM file with the same
 * contents as a specified input file, only that dates are changed 
 * or added based on a specified database.
 *
 * @author Vegard Brox
 */
public class GEDCOMOutput
{
   // Private state constants
   private static final int STATE_0          = 0;
   private static final int STATE_1          = 1;
   private static final int STATE_2          = 2;
   private static final int STATE_WRITE      = 8;
   private static final int STATE_IGNORE     = 9;
   private static final int STATE_FINISHED   = -1;
   
   // Private mode constants
   private static final int MODE_NONE        = 0;
   private static final int MODE_INDIV       = 1;
   private static final int MODE_FAMILY      = 2;

   // Private variables
   private BufferedReader myInput;
   private PrintWriter myOutput;
   private Hashtable myPeople;
   private Hashtable myFamilies;
   private Person myPerson;
   private Family myFamily;
   private YearRange myDate;
   private String myDateType;
   private boolean myBirthFound = false;
   private boolean myDeathFound = false;
   private boolean myChristeningFound = false;
   private boolean myBurialFound = false;
   private int myState;
   private int myMode;
   private boolean myKeepDates = false;

   /**
    * Constructs a new GEDCOMOutput object for the specified 
    * database and files. Included header information in output file.
    *
    * @param inputFile The original GEDCOM file
    * @param outputFile The new GEDCOM file (will be overwritten if it exists)
    * @param database The database to get estimated dates from
    * @exception IOException If one of the files are not accessible
    */
   public GEDCOMOutput( File inputFile, File outputFile, GenealogyDB database )
      throws IOException
   {
      // Create reader/writer
      myInput = new BufferedReader( new FileReader( inputFile ) );
      myOutput = new PrintWriter( new FileWriter( outputFile ) );
      
      // Get hashtables
      myPeople = database.getPeople();
      myFamilies = database.getFamilies();
      
      // Set initial state and mode
      myState = STATE_0;
      myMode = MODE_NONE;
   }

   /**
    * Constructs a new GEDCOMOutput object for the specified 
    * database and files.
    *
    * @param inputFile The original GEDCOM file
    * @param outputFile The new GEDCOM file (will be overwritten if it exists)
    * @param database The database to get estimated dates from
    * @param keepDates True if the dates from the input file should be written
    *        unchanged to output
    * @exception IOException If one of the files are not accessible
    */
   public GEDCOMOutput( File inputFile, File outputFile, GenealogyDB database, 
                        boolean keepDates ) throws IOException
   {
      this( inputFile, outputFile, database );
      myKeepDates = keepDates;
      if( keepDates )
      {
         myState = STATE_IGNORE;
      }
   }
   
   
   /**
    * Performs the actual operation of writing the new file.
    *
    * @exception IOException If there is a problem with 
    *   reading or writing files
    */
   public void output() throws IOException
   {
      // Check that the state is OK
      if( myState == STATE_FINISHED )
      {
         throw new IllegalStateException( "Output has already been written" );
      }
   
      // Loop through all input lines
      String line = myInput.readLine();
      while( line != null )
      {
         // Check the line, then write it
         Enumeration newLines = processLine( line );
         while( newLines.hasMoreElements() )
         {
            myOutput.println( (String)newLines.nextElement() );
         }
         
         // Read next line
         line = myInput.readLine();
      }
      
      // Flush and close reader/writer and set state
      myInput.close();
      myOutput.flush();
      myOutput.close();
      myState = STATE_FINISHED;
   }
   
   /**
    * Processes one single line from the input file. This processing works
    * as a state machine, because what action to take at a given point
    * depends on the earlier lines. 
    *
    * @param line The line to process
    * @return An enumeration of lines to insert in the output file
    */
   private Enumeration processLine( String line )
   {
      Vector linesToAdd = null;
   
      // Action to take depends on state and mode
      // If we are in state 2 we are looking for a date tag
      if( myState == STATE_2 )
      {
         // Check if we should go to state 0
         if( line.length() > 0 && line.charAt( 0 ) == '0' )
         {
            // If this is a person - insert events that weren't specified
            // in the original file
            if( myMode == MODE_INDIV )
            {
               linesToAdd = checkMissingDates( true );
            }
            
            // Reset state etc.
            myState = STATE_0;
            myMode = MODE_NONE;
            myBirthFound = false;
            myChristeningFound = false;
            myDeathFound = false;
            myBurialFound = false;
         }
         // Check if we should go to state 1
         else if( line.length() > 1 && line.charAt( 0 ) == '1' )
         {
            // If this is a person, the last read event did not have
            // a date tag - insert it
            if( myMode == MODE_INDIV )
            {
               linesToAdd = checkMissingDate();
            }
            myState = STATE_1;
         }
         else
         {
            // If this line is the date - encode it and go to state 1
            String elementType = GEDCOMLocater.getElementType( line );
            if( elementType.equals( GEDCOMLocater.DATE ) )
            {
               line = encodeDate( myDate );
               myState = STATE_1;
            }
         }
      }
      
      // If we are in state 1, we are looking for an event tag (e.g. birth)
      if( myState == STATE_1 )
      {
         // Check if we should go to state 0
         if( line.length() > 0 && line.charAt( 0 ) == '0' )
         {
            // If this is a person - insert events that weren't specified
            // in the original file
            if( myMode == MODE_INDIV )
            {
               linesToAdd = checkMissingDates( false );
            }
            
            // Reset state etc.
            myState = STATE_0;
            myMode = MODE_NONE;
            myBirthFound = false;
            myChristeningFound = false;
            myDeathFound = false;
            myBurialFound = false;
         }
         // If this is a person - look for person events
         else if( myMode == MODE_INDIV )
         {
            // Look for birth, christening, death and burial.
            // When found: if the date has been changed by the program
            //  we go to state 2 to search for the date
            //  Otherwise we just mark the date as found and continue
         
            String elementType = GEDCOMLocater.getElementType( line );
            if( elementType.equals( GEDCOMLocater.BIRTH ) )
            {
               YearRange birth = myPerson.getBirth();
               if( birth.isChanged() )
               {
                  myDate = birth;
                  myDateType = GEDCOMLocater.BIRTH;
                  myState = STATE_2;
               }
               myBirthFound = true;
            }
            else if( elementType.equals( GEDCOMLocater.CHRISTENING ) )
            {
               YearRange christening = myPerson.getChristening();
               if( christening.isChanged() )
               {
                  myDate = christening;
                  myDateType = GEDCOMLocater.CHRISTENING;
                  myState = STATE_2;
               }
               myChristeningFound = true;
            }
            else if( elementType.equals( GEDCOMLocater.DEATH ) )
            {
               YearRange death = myPerson.getDeath();
               if( death.isChanged() )
               {
                  myDate = death;
                  myDateType = GEDCOMLocater.DEATH;
                  myState = STATE_2;
               }
               myDeathFound = true;
            }
            else if( elementType.equals( GEDCOMLocater.BURIAL ) )
            {
               YearRange burial = myPerson.getBurial();
               if( burial.isChanged() )
               {
                  myDate = burial;
                  myDateType = GEDCOMLocater.BURIAL;
                  myState = STATE_2;
               }
               myBurialFound = true;
            }
         }
         // If this is a family - search for family events
         else if( myMode == MODE_FAMILY )
         {
            // Search for marriage
            // When found: if date has been changed by program, go to
            //  state 2 to search for date
            String elementType = GEDCOMLocater.getElementType( line );
            if( elementType.equals( GEDCOMLocater.MARRIAGE ) )
            {
               YearRange marriage = myFamily.getMarriage();
               if( marriage.isChanged() )
               {
                  myDate = marriage;
                  myState = STATE_2;
               }
            }
         }
      }
      
      // If we are in state 0, we are searching for individual and
      // family records
      if( myState == STATE_0 || 
          myState == STATE_WRITE || 
          myState == STATE_IGNORE )
      {
         // Check if this line is a possible start for 
         //  individual or family record
         if( line != null && line.length() > 0 && line.charAt( 0 ) == '0' )
         {
            // Check if this is the start of an individual record 
            if( line.indexOf( GEDCOMLocater.INDIVIDUAL ) > 0 )
            {
               // Get this person from the database
               String id = GEDCOMInterpreter.interpretID( line );
               Person person = (Person)myPeople.get( id );
               if( person != null )
               {
                  // Store person for later use
                  myPerson = person;
          
                  // Set state and mode
                  if( myKeepDates )
                  {
                     myState = STATE_WRITE;
                  }
                  else
                  {
                     myState = STATE_1;
                     myMode = MODE_INDIV;
                  }
               }
               else
               {
                  // This person is not in DB - ignore
                  myState = STATE_IGNORE;
               }
            }
            // Check if this line is the start of a family record
            else if( line.indexOf( GEDCOMLocater.FAMILY ) > 0 )
            {
               // Get this family from the database
               String id = GEDCOMInterpreter.interpretID( line );
               Family family = (Family)myFamilies.get( id );
               if( family != null )
               {
                  // Store the family for later use
                  myFamily = family;
                  
                  // Set state and mode
                  if( myKeepDates )
                  {
                     myState = STATE_WRITE;
                  }
                  else
                  {
                     myState = STATE_1;
                     myMode = MODE_FAMILY;
                  }
               }
               else
               {
                  // This family is not in DB - ignore
                  myState = STATE_IGNORE;
               }
            }
            else
            {
               // Start of header information or similar
               // Ignore or not, depending on boolean value
               myState = (myKeepDates)? STATE_IGNORE : STATE_0;
            }
         }
      } 
      
      // If no extra lines to insert - make vector
      if( linesToAdd == null )
      {
         linesToAdd = new Vector();
      }

      // Add "this" line unless in ignore mode, then return
      if( myState != STATE_IGNORE )
      {
         linesToAdd.addElement( line );
      }
      return linesToAdd.elements();
   }

   /**
    * Encodes a GEDCOM line based on a date
    *
    * @param date The date to encode for, given as a YearRange
    * @return A string with GEDCOM code for a the given date
    */
   private String encodeDate( YearRange date )
   {
      // Encode as single year or range
      if( date.getStart() == date.getEnd() )
      {
         return "2 DATE " + date.getStart();
      }
      else
      {
         return "2 DATE BET " + date.getStart() + 
                " AND " + date.getEnd();
      }
   }
   
   /**
    * Checks if any events have not been specified for an individual,
    * and if necessary creates the lines of GEDCOM code that should
    * be added for these events
    *
    * @param searchForSingleDate Should be true if we are about to go 
    *  straight from state 2 to state 0 (i.e. an event has been 
    *  specified without a date), false otherwise
    */
   private Vector checkMissingDates( boolean searchForSingleDate )
   {
      Vector toAdd;
      
      // Search for single date if necessary
      if( searchForSingleDate )
      {
         toAdd = checkMissingDate();
      }
      else
      {
         toAdd = new Vector();
      }
      
      // Add lines for each eventtype if they weren't already there
      if( !myBirthFound )
      {
         toAdd.addElement( "1 " + GEDCOMLocater.BIRTH ); 
         toAdd.addElement( encodeDate( myPerson.getBirth() ) );
      }
      if( !myChristeningFound )
      {
         toAdd.addElement( "1 " + GEDCOMLocater.CHRISTENING );
         toAdd.addElement( encodeDate( myPerson.getChristening() ) );
      }
      if( !myDeathFound )
      {
         toAdd.addElement( "1 " + GEDCOMLocater.DEATH );
         toAdd.addElement( encodeDate( myPerson.getDeath() ) );
      }
      if( !myBurialFound )
      {
         toAdd.addElement( "1 " + GEDCOMLocater.BURIAL );
         toAdd.addElement( encodeDate( myPerson.getBurial() ) );
      }
      return toAdd;
   }
   
   /**
    * Creates a line of GEDCOM code containing a date. Should be 
    * called if an event for a person has been specified without
    * any specified date at all.
    *
    * @return A vector with the line of GEDCOM code
    */
   private Vector checkMissingDate()
   {
      Vector toAdd = new Vector();
      
      // Add line depending on what event we are currently reading
      if( myDateType.equals( GEDCOMLocater.BIRTH ) )
      {
         toAdd.addElement( encodeDate( myPerson.getBirth() ) );
      }
      else if( myDateType.equals( GEDCOMLocater.CHRISTENING ) )
      {
         toAdd.addElement( encodeDate( myPerson.getChristening() ) );
      }
      else if( myDateType.equals( GEDCOMLocater.DEATH ) )
      {
         toAdd.addElement( encodeDate( myPerson.getDeath() ) );
      }
      else if( myDateType.equals( GEDCOMLocater.BURIAL ) )
      {
         toAdd.addElement( encodeDate( myPerson.getBurial() ) );
      }
      return toAdd;
   }
}

