// File: GEDCOMLocater.java

// Import
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * This class locates and picks out relevant lines from a file of GEDCOM code.
 * The lines are then passed on to the GEDCOMInterpreter class and 
 * initialised person or family objects are returned to the user.
 * 
 * @author Vegard Brox
 */
public class GEDCOMLocater
{
   // CONSTANTS
   /** GEDCOM code for individual */
   public static final String INDIVIDUAL     = "INDI";
   /** GEDCOM code for family */
   public static final String FAMILY         = "FAM";
   /** GEDCOM code for birth */
   public static final String BIRTH          = "BIRT";
   /** GEDCOM code for death */
   public static final String DEATH          = "DEAT";
   /** GEDCOM code for christening */
   public static final String CHRISTENING    = "CHR";
   /** GEDCOM code for burial */
   public static final String BURIAL         = "BURI";
   /** GEDCOM code for sex */
   public static final String SEX            = "SEX";
   /** GEDCOM code for date */
   public static final String DATE           = "DATE";
   /** GEDCOM code for family where this person was child */
   public static final String CFAMILY        = "FAMC";
   /** GEDCOM code for family where this person was spouse */
   public static final String SFAMILY        = "FAMS";
   /** GEDCOM code for husband */
   public static final String HUSBAND        = "HUSB";
   /** GEDCOM code for wife */
   public static final String WIFE           = "WIFE";
   /** GEDCOM code for child */
   public static final String CHILD          = "CHIL";
   /** GEDCOM code for marriage */
   public static final String MARRIAGE       = "MARR";
   /** GEDCOM code for trailer (end of file) */
   public static final String TRAILER        = "TRLR";
   
   private static final int INDMODE = 1;
   private static final int FAMMODE = 2;
   private static final int READ_AHEAD_LIMIT = 100;

   // VARIABLES
   private BufferedReader myIndiv   = null;
   private BufferedReader myFamily  = null;
   private String myID              = null;
   private String myBirth           = null;
   private String myDeath           = null;
   private String myChristening     = null;
   private String myBurial          = null;
   private String mySex             = null;
   private String myCFamily         = null;
   private Vector mySFamilies       = null;
   
   private String myFID             = null;
   private String myMarriage        = null;
   private String myHusband         = null;
   private String myWife            = null;
   private Vector myChildren        = null;
   private boolean myIsMarriage     = false;

   
   /**
    * Constructor that prepares for reading from the specified file.
    * 
    * @param file The GEDCOM file to read from
    * @exception FileNotFoundException If the specified file could not be found
    */
   public GEDCOMLocater( File file ) throws FileNotFoundException
   {
      // Initialise two buddered readers to be able to read individuals and 
      //  families independent
      myIndiv = new BufferedReader( new FileReader( file ) );
      myFamily = new BufferedReader( new FileReader( file ) );
   }
   
   /**
    * Reads information for the next person in the GEDCOM file and returns it
    * as a person object. If no more persons exist, null is returned.
    *
    * @return A person object for the next individual in the file
    */
   public Person nextPerson()
   {
      // Reset variables
      myID = null;
      myBirth = null;
      myDeath = null;
      myChristening = null;
      myBurial = null;
      mySex = null;
      myCFamily = null;
      mySFamilies = new Vector();
      
      try
      {
         // Try to locate information
         locate( INDMODE );
      }
      catch( IOException e )
      {
         // Return null if something went wrong
         return null;
      }
      
      // Return null if no ID was found
      if( myID == null )
      {
         return null;
      }
      
      // Interpret the data and return a person object
      Person person = GEDCOMInterpreter.person( myID, myBirth, myDeath, 
                                          myChristening, myBurial, mySex,
                                          myCFamily, mySFamilies );
      return person;
   }
   
   /**
    * Reads information for the next family in the GEDCOM file and returns it
    * as a family object. If no more families exist, null is returned.
    *
    * @return A family object for the next family in the file
    */
   public Family nextFamily()
   {
      // Reset variables
      myFID       = null;
      myMarriage  = null;
      myHusband   = null;
      myWife      = null;
      myChildren  = new Vector();
      myIsMarriage = false;
      
      try
      {
         // Try to locate the family data
         locate( FAMMODE );
      }
      catch( IOException e )
      {
         // Return null if something went wrong
         return null;
      }
      
      // If no ID was found, return null
      if( myFID == null )
      {
         return null;
      }
      
      // Interpret the data and return family object
      Family family = GEDCOMInterpreter.family( myFID, myIsMarriage, myMarriage, 
                                                myHusband, myWife, myChildren );
      return family;
   }
   
   
   /**
    * Locates data for next individual or family in the file and stores
    * the data in member variables.
    * 
    * @param mode Either individual (INDMODE) or family (FAMMODE)
    * @exception IOException If something went wrong when reading the file
    */
   private void locate( int mode ) throws IOException
   {
      String strLine = null;
      BufferedReader in = null;
      
      // If individual mode
      if( mode == INDMODE )
      {
         // Use correct reader
         in = myIndiv;
         
         // Skip lines until next individual is found
         do
         {
            strLine = in.readLine();
         } while( strLine != null && 
                  !( strLine.length() > 1 &&
                     strLine.charAt( 0 ) == '0' &&
                     ( strLine.indexOf( INDIVIDUAL ) > 0 ||
                       strLine.indexOf( TRAILER ) > 0 ) ) );
         
         // Exit method if end of file
         if( strLine == null || strLine.indexOf( TRAILER ) > 0 )
         {
            return;
         }
                    
         // Store this line (as it contains the ID) and read next
         myID = strLine;
         strLine = in.readLine();
         
         // While still reading data for this person
         while( strLine != null && 
                ( strLine.length() < 1 || strLine.charAt( 0 ) != '0' ) )
         {
            // Mark current position
            in.mark( READ_AHEAD_LIMIT );
         
            // Get element type for this line
            String elementType = getElementType( strLine );
            
            // If no element type
            if( elementType.equals( "" ) )
            {
            }
            // If element type is birth - locate relevant date
            else if( elementType.equals( BIRTH ) )
            {
               myBirth = locateDate( in );
            }
            // If element type is death - locate relevant date
            else if( elementType.equals( DEATH ) )
            {
               myDeath = locateDate( in );
            }
            // If element type is christening - locate relevant date
            else if( elementType.equals( CHRISTENING ) )
            {
               myChristening = locateDate( in );
            }
            // If element type is burial - locate relevant date
            else if( elementType.equals( BURIAL ) )
            {
               myBurial = locateDate( in );
            }
            // If element type is sex - store this line
            else if( elementType.equals( SEX ) )
            {
               mySex = strLine;
            }
            // If element type is child family - store this line
            else if( elementType.equals( CFAMILY ) )
            {
               myCFamily = strLine;
            }
            // If element type is spouse family - add this line to vector
            else if( elementType.equals( SFAMILY ) )
            {
               mySFamilies.addElement( strLine );
            }
            
            // Read new line
            strLine = in.readLine();
         }
         
         // Reset, as we have read one line to much
         if( strLine != null )
         {
            in.reset();
         }
      }
      // If family mode
      else
      {
         // Use correct reader
         in = myFamily;
         
         // Skip lines up to next family record
         do
         {
            strLine = in.readLine();
         } while( strLine != null && 
                  !( strLine.length() > 1 &&
                     strLine.charAt( 0 ) == '0' &&
                     ( strLine.indexOf( FAMILY ) > 0 || 
                       strLine.indexOf( TRAILER ) > 0  ) ) );
         
         
         // Exit method if end of file
         if( strLine == null || strLine.indexOf( TRAILER ) > 0 )
         {
            return;
         }
         
         
         // Store this line (as it contains the ID) and read next line
         myFID = strLine;
         strLine = in.readLine();
         
         // While we are still reading this family record
         while( strLine != null && 
                ( strLine.length() < 1 || strLine.charAt( 0 ) != '0' ) )
         {
            in.mark( READ_AHEAD_LIMIT );

            // Get the element type
            String elementType = getElementType( strLine );
            
            // If no element type - do not read new line
            if( elementType.equals( "" ) )
            {
            }
            // If element type is marriage - locate relevant date
            else if( elementType.equals( MARRIAGE ) )
            {
               myIsMarriage = true;
               myMarriage = locateDate( in );
            }
            // If element type is husband - store this line
            else if( elementType.equals( HUSBAND ) )
            {
               myHusband = strLine;
            }
            // If element type is wife - store this line
            else if( elementType.equals( WIFE ) )
            {
               myWife = strLine;
            }
            // If element type is child - add this line to vector
            else if( elementType.equals( CHILD ) )
            {
               myChildren.addElement( strLine );
            }

            // Read new line
            strLine = in.readLine();
         }
         
         // Reset as we have read one line too much
         if( strLine != null )
         {
            in.reset();
         }
      }
   }
   
   /**
    * Locates the date for an event in the file. 
    *
    * @param in A buffered reader for the GEDCOM file located at the starting 
    *           point for the date search
    * @return A line of GEDCOM code containing the date for the event, or
    *         null if no date could be found
    * @exception IOException If there was a problem reading from the file
    */
   private String locateDate( BufferedReader in ) throws IOException
   {
      String strLine = null;
      
      // Read lines while the date is not found and we are still reading
      //  lines for this event
      do
      {
         in.mark( READ_AHEAD_LIMIT );
         strLine = in.readLine();
      }
      while( strLine != null &&
             ( strLine.length() < 1 ||
               ( strLine.charAt( 0 ) == '2' && 
                 strLine.indexOf( DATE ) < 0 ) ) );
   
      // Return the last line read if it contains the date
      if( strLine != null && strLine.indexOf( DATE ) > 0 )
      {
         return strLine;
      }
      // Otherwise return null
      in.reset();
      return null;
   }
   
   /**
    * Finds the element type from a line of GEDCOM code. 
    * The element type is e.g. birth, death, wife, child.
    *
    * @param line The line of GEDCOM code
    * @return The element type for this line, or an empty string
    *         if no element type could be found
    */
   public static String getElementType( String line )
   {
      // Split into tokens
      StringTokenizer tokenizer = new StringTokenizer( line, " " );
      // Try to reach and return the second token
      if( tokenizer.hasMoreTokens() )
      {
         tokenizer.nextToken();
         if( tokenizer.hasMoreTokens() )
         {
            return tokenizer.nextToken();
         }
      }
      // If unsuccessful - return empty string
      return "";
   }
}
