// File: GEDCOMInterpreter.java

// Import
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.NoSuchElementException;


/**
 * This class interprets the meaning of given strings containing GEDCOM code.
 * It assumes that only the necessary lines from a GEDCOM file are already
 * selected, as done by the GEDCOMLocater class.
 * 
 * @author Vegard Brox
 */
public class GEDCOMInterpreter
{
   // Constants used in date recognititon
   private static final String ABOUT   = "ABT";
   private static final String BEFORE  = "BEF";
   private static final String AFTER   = "AFT";
   private static final String BETWEEN = "BET";
   private static final String AND     = "AND";
   private static final String FEMALE  = "F";


   /**
    * Interprets the given strings related to a person and creates a
    * person object.
    *
    * @param id A string containing the ID of the person
    * @param birth A string containing the birth date of the person
    * @param death A string containing the death date of the person
    * @param christening A string containing the christening date of the person
    * @param burial A string containing the burial date of the person
    * @param sex A string containing the sex of the person
    * @param cFamily A string containing the ID of the family where 
    *                this person is a child
    * @param sFamilies A vector of strings containing IDs of families where
    *                this person is a spouse
    * @return A person object initialised with the appropriate values
    */
   public static Person person( String id, String birth, String death,
                                String christening, String burial, String sex,
                                String cFamily, Vector sFamilies )
   {
      // Interpret information
      String intpID = interpretID( id );
      YearRange intpBirth = interpretDate( birth );
      YearRange intpDeath = interpretDate( death );
      YearRange intpChristening = interpretDate( christening );
      YearRange intpBurial = interpretDate( burial );
      int intpSex = interpretSex( sex );
      String intpCFamily = interpretReference( cFamily );
      
      Vector intpSFamilies = new Vector();
      for( int i=0; i < sFamilies.size(); i++ )
      {
         intpSFamilies.addElement( interpretReference( 
                        (String) sFamilies.elementAt( i ) ) );
      }
      
      // Create person object and return it
      Person person = new Person( intpID, intpBirth, intpDeath,
                                  intpChristening, intpBurial, intpSex,
                                  intpCFamily, intpSFamilies );
      return person;
   }
   
   /**
    * Interprets the given strings related to a family and creates
    * a family object.
    * 
    * @param id A string containing the ID of the family
    * @param isMarriage True if there actually was a marriage between
    *                   husband and wife in this family, false otherwise
    * @param marriage A string containing the marriage date for the 
    *                 husband and wife in the family
    * @param husband A string containing the ID of the husband
    * @param wife A string containing the ID of the wife
    * @param children A vector of strings containing the IDs of the children
    * @return A family object initialised with appropriate values
    */
   public static Family family( String id, boolean isMarriage, String marriage, 
                                String husband, String wife, Vector children )
   {
      // Interpret values
      String      intpID         = interpretID( id );
      String      intpHusband    = interpretReference( husband );
      String      intpWife       = interpretReference( wife );
      YearRange   intpMarriage   = null;
      if( isMarriage )
      {
         intpMarriage = interpretDate( marriage );
      }
      
      Vector intpChildren = new Vector();
      for( int i=0; i < children.size(); i++ )
      {
         intpChildren.addElement( interpretReference( 
                        (String) children.elementAt( i ) ) );
      }
      
      // Create family object and return it
      Family family = new Family( intpID, intpMarriage, intpHusband, 
                                  intpWife, intpChildren );
      return family;
   }
   
   /**
    * Interprets an ID from a string containing a line of GEDCOM code
    * 
    * @param idString The line of GEDCOM code
    * @return The ID
    */
   public static String interpretID( String idString )
   {
      // Split string into tokens
      StringTokenizer tokenizer = new StringTokenizer( idString, " " );
      
      // Try to reach second token
      if( tokenizer.hasMoreTokens() )
      {
         tokenizer.nextToken();
         if( tokenizer.hasMoreTokens() )
         {
            // Return the token with the ID
            return tokenizer.nextToken();
         }
      }
      // Default return if something went wrong
      return null;
   }
   
   /**
    * Interprets sex from a string containing a line of GEDCOM code.
    * If null is given as parameter or any problem occurs in interpretation,
    * the person is assumed to be male, as the heuristics are least 
    * restrictive for males.
    * 
    * @param sexString The line of GEDCOM code
    * @return Either Person.MALE or Person.FEMALE
    * @see Person
    */
   private static int interpretSex( String sexString )
   {
      // If no inputstring - assume male 
      if( sexString == null )
      {
         return Person.MALE;
      }
      
      StringTokenizer tokenizer = new StringTokenizer( sexString, " " );
      try
      {
         // Ignore first two tokens (e.g. "1 SEX")
         tokenizer.nextToken();
         tokenizer.nextToken();
         
         // Read token containing sex - return correct value
         String token = tokenizer.nextToken();
         if( token.equals( FEMALE ) )
         {
            return Person.FEMALE;
         }
         else
         {
            return Person.MALE;
         }
      }
      catch( NoSuchElementException e )
      {
         // If something wrong - assume male
         return Person.MALE;
      }
   }
   
   /**
    * Interprets a reference to another individual or family from a string
    * containing a line of GEDCOM code.
    * 
    * @param referenceString The line of GEDCOM code
    * @return The ID from the parameter string
    */
   private static String interpretReference( String referenceString )
   {
      // If no string specified - return null
      if( referenceString == null )
      {
         return null;
      }
      
      // Split up in tokens
      StringTokenizer tokenizer = new StringTokenizer( referenceString, " " );
      
      try
      {
         // Ignore first two tokens (e.g. "1 FAMC")
         tokenizer.nextToken();
         tokenizer.nextToken();
         
         // Return third token unchanged
         return tokenizer.nextToken();
      }
      catch( NoSuchElementException e )
      {
         // Something went wrong - return null
         return null;
      }
   }
   
   /**
    * Interprets a date from a string containing a line of GEDCOM code.
    * 
    * @param dateString The line of GEDCOM code
    * @return A year range representing the date from the parameter string
    */
   private static YearRange interpretDate( String dateString )
   {
      // If no inputstring - return default year range
      if( dateString == null )
      {
         return new YearRange();
      }
    
      // Split string into tokens
      StringTokenizer tokenizer = new StringTokenizer( dateString, " " );
      try
      {
         // Ignore first two tokens (e.g. "2 DATE")
         tokenizer.nextToken();
         tokenizer.nextToken();
      }
      catch( NoSuchElementException e )
      {
         // Tokens not present
         return new YearRange();
      }

      int year = 0;
      int year2 = 9999;
      String mode = null;
      
      // We have now reached the date in the string
      while( tokenizer.hasMoreTokens() )
      {
         String token = tokenizer.nextToken();
         
         // Check if any special mode should be enabled
         if( token.equals( ABOUT ) || token.equals( BEFORE ) || 
             token.equals( AFTER ) || token.equals( BETWEEN ) )
         {
            mode = token;
         }
         else
         {
            try
            {
               // Check if this is a year range of old style form xxxx/yyyy
               int slashIndex = token.indexOf( '/' );
               if( slashIndex > 0 )
               {
                  String tokenA = token.substring( 0, slashIndex );
                  String tokenB = token.substring( slashIndex + 1 );
                  int numberA = Integer.parseInt( tokenA );
                  int numberB = Integer.parseInt( tokenB );
                  
                  if( numberA > 31 && numberA < 9999 && 
                      numberB > 31 && numberB < 9999 )
                  {
                     year = numberA;
                     year2 = numberB;
                     mode = BETWEEN;
                  }
               }
               else
               {
                  // Try to convert to a number
                  int number = Integer.parseInt( token );
                  if( number > 31 )
                  {
                     // If reading second year in a range - store in year2
                     if( mode != null && mode.equals( BETWEEN ) && year > 0 )
                     {
                        year2 = number;
                     }
                     // Any other year is stored in the year variable
                     else
                     {
                        year = number;
                     }
                  }
               }
            }
            catch( NumberFormatException e )
            {
               // Ignore
            }
         }
      }
      
      // Initialise a year range depending on mode and year from file
      YearRange range = null;
      if( mode == null )
      {
         if( year != 0 )
         {
            range = new YearRange( year, year );
         }
         else 
         {
            range = new YearRange();
         }
      }
      else if( mode.equals( AFTER ) )
      {
         range = new YearRange( YearRange.AFTER, year );
      }
      else if( mode.equals( BEFORE ) )
      {
         range = new YearRange( YearRange.BEFORE, year );
      }
      else if( mode.equals( ABOUT ) )
      {
         EstimationProperties properties = EstimationProperties.getInstance();
         int offset = properties.getProperty( "aboutValue" );
         range = new YearRange( year-offset, year+offset );
      }
      else if( mode.equals( BETWEEN ) )
      {
         range = new YearRange( year, year2 );
      }
      
      // Return the year range
      return range;
   }
}
