// File: Estimator.java

//Import
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;


/**
 * This class estimates missing/unspecific dates in a given database.
 *
 * @author Vegard Brox
 */
public class Estimator
{
   // Private constants
   private static final int MAX_ITERATIONS   = 100;
   private static final String BIRTH         = "BIRTH";
   private static final String CHRISTENING   = "CHRISTENING";
   private static final String DEATH         = "DEATH";
   private static final String BURIAL        = "BURIAL";
   private static final String MARRIAGE      = "MARRIAGE";
   

   // Private variables
   private GenealogyDB myDB;
   private Hashtable myLog;
   private Vector myErrorDBs;
   
   /**
    * Creates a new estimator object for the specified database.
    * @param database the database to perform estimation on
    */
   public Estimator( GenealogyDB database )
   {
      myDB = database;
      myLog = new Hashtable();
      myErrorDBs = new Vector();
   }

   /**
    * Loops iteratively through the specified database, and narrows
    * year ranges where possible, using a number of logical and 
    * heuristical constraints.
    *
    * @return The number of iterations required to do the estimation
    */
   public int estimate() 
   {
      // Get the estimation properties
      EstimationProperties properties = EstimationProperties.getInstance();
      int maxLivingAge     = properties.getProperty( "maxLivingAge" );
      int maxChristening   = properties.getProperty( "maxChristening" );
      int maxBurial        = properties.getProperty( "maxBurial" );
      int minChildBirth    = properties.getProperty( "minChildBirth" );
      int maxChildBirth    = properties.getProperty( "maxChildBirth" );
      int minMarriage      = properties.getProperty( "minMarriage" );
      
      // Get hashtables with people and families
      Hashtable people = myDB.getPeople();
      Hashtable families = myDB.getFamilies();
      
      // Loop while more year ranges to narrow
      int iterationCount = 0;
      boolean changed = true;
      while( changed && iterationCount++ < MAX_ITERATIONS )
      {
         // Default is not to continue
         changed = false;
      
         // Get people IDs and loop through all of them
         Enumeration peopleIDs = people.keys();
         while( peopleIDs.hasMoreElements() )
         {
            // Get a person and data about the person
            String personID = (String)peopleIDs.nextElement();
            Person person = (Person)people.get( personID );
            YearRange birth = person.getBirth();
            YearRange christening = person.getChristening();
            YearRange death = person.getDeath();
            YearRange burial = person.getBurial();
       
            boolean c1, c2, c3, c4, c5, c6, c7;
       
            // Constraint 1 (DD >= BD)
            c1 = applyConstraint( death, birth, 0, 
                                  personID, DEATH, personID, BIRTH );
            // Constraint 2 (DD >= ChD)
            c2 = applyConstraint( death, christening, 0,
                                  personID, DEATH, personID, CHRISTENING );  
            // Constraint 3 (BD >= DD - maxLivingAge)
            c3 = applyConstraint( birth, death, -maxLivingAge,
                                  personID, BIRTH, personID, DEATH );
            // Constraint 4 (ChD >= BD)
            c4 = applyConstraint( christening, birth, 0,
                                  personID, CHRISTENING, personID, BIRTH );
            // Constraint 5 (BD >= ChD - maxChristening)
            c5 = applyConstraint( birth, christening, -maxChristening,
                                  personID, BIRTH, personID, CHRISTENING );
            // Constraint 6 (BuD >= DD)
            c6 = applyConstraint( burial, death, 0,
                                  personID, BURIAL, personID, DEATH );
            // Constraint 7 (DD >= BuD - maxBurial)
            c7 = applyConstraint( death, burial, -maxBurial,
                                  personID, DEATH, personID, BURIAL );
            
            if( c1 || c2 || c3 || c4 || c5 || c6 || c7 )
            {
               changed = true;
            }

            // Check for negative year ranges
            checkYearRange( birth, person.getID(), BIRTH );
            checkYearRange( christening, person.getID(), CHRISTENING );
            checkYearRange( death, person.getID(), DEATH );
            checkYearRange( burial, person.getID(), BURIAL );

            // Get father of the person
            Person father = myDB.getFather( personID );
            if( father != null )
            {
               YearRange birthFather = father.getBirth();
               YearRange deathFather = father.getDeath();
               boolean c10, c11;
            
               // Constraint 10 (BD >= FBD + minChildBirth)
               c10 = applyConstraintForward( 
                        birth, birthFather, minChildBirth,
                        personID, BIRTH, father.getID(), BIRTH );
               // Constraint 11 (FDD >= BD - 1)
               c11 = applyConstraintBackward( deathFather, birth, -1,
                        father.getID(), DEATH, personID, BIRTH );
               if( c10 || c11 )
               {
                  changed = true;
               }
               
               // Check year range
               checkYearRange( birth, person.getID(), BIRTH );
            }
            
            // Get mother of the person
            Person mother = myDB.getMother( personID );
            if( mother != null )
            {
               YearRange birthMother = mother.getBirth();
               YearRange deathMother = mother.getDeath();
               boolean c12, c13, c14;
               
               // Constraint 12 (BD >= MBD + minChildBirth)
               c12 = applyConstraintForward( 
                        birth, birthMother, minChildBirth,
                        personID, BIRTH, mother.getID(), BIRTH );
               // Constraint 13 (MDD >= BD)
               c13 = applyConstraintBackward( deathMother, birth, 0,
                        mother.getID(), DEATH, personID, BIRTH );
               // Constraint 14 (MBD >= BD - maxChildBirth)
               c14 = applyConstraintBackward( 
                        birthMother, birth, -maxChildBirth,
                        mother.getID(), BIRTH, personID, BIRTH );
               if( c12 || c13 || c14 )
               {
                  changed = true;
               }
               
               // Check year range
               checkYearRange( birth, person.getID(), BIRTH );
            }
            
            // Get families where individual is spouse
            Enumeration sFamilies = myDB.getSFamilies( personID );
            while( sFamilies.hasMoreElements() )
            {
               Family sFamily = (Family)sFamilies.nextElement();
               YearRange marriage = sFamily.getMarriage();
               boolean c8, c9;
               
               if( marriage != null )
               {
                  // Constraint 8 (DD >= MD)
                  c8 = applyConstraintForward( death, marriage, 0,
                           personID, DEATH, sFamily.getID(), MARRIAGE );
                  // Constraint 9 (MD >= BD + minMarriage)
                  c9 = applyConstraintBackward( marriage, birth, minMarriage,
                           sFamily.getID(), MARRIAGE, personID, BIRTH );
                  if( c8 || c9 )
                  {
                     changed = true;
                  }
                  
                  // Check year ranges
                  checkYearRange( birth, person.getID(), BIRTH );
                  checkYearRange( death, person.getID(), DEATH );
               }
               
               // Get children in family
               Enumeration children = myDB.getChildren( sFamily.getID() );
               while( children.hasMoreElements() )
               {
                  Person child = (Person)children.nextElement();
                  YearRange birthChild = child.getBirth();
                  
                  // Apply different constraints depending on whether current
                  //  individual is male or female
                  if( person.isMale() )
                  {
                     boolean c10, c11;
                     // Constraint 10 (BD >= FBD + minChildBirth)
                     c10 = applyConstraintBackward( 
                              birthChild, birth, minChildBirth,
                              child.getID(), BIRTH, personID, BIRTH );
                     // Constraint 11 (FDD >= BD - 1)
                     c11 = applyConstraintForward( death, birthChild, -1,
                              personID, DEATH, child.getID(), BIRTH );
                     if( c10 || c11 )
                     {
                        changed = true;
                     }
                  }
                  else
                  {
                     boolean c12, c13, c14;
                     // Constraint 12 (BD >= MBD + minChildBirth)
                     c12 = applyConstraintBackward( 
                              birthChild, birth, minChildBirth,
                              child.getID(), BIRTH, personID, BIRTH );
                     // Constraint 13 (MDD >= BD)
                     c13 = applyConstraintForward( death, birthChild, 0,
                              personID, DEATH, child.getID(), BIRTH );
                     // Constraint 14 (MBD >= BD - maxChildBirth)
                     c14 = applyConstraintForward( 
                              birth, birthChild, -maxChildBirth,
                              personID, BIRTH, child.getID(), BIRTH );
                     if( c12 || c13 || c14 )
                     {
                        changed = true;
                     }
                  }
                  // Check year ranges
                  checkYearRange( birth, person.getID(), BIRTH );
                  checkYearRange( death, person.getID(), DEATH );
               }
            }
         } // end while more people
         
         // Loop through all families
         Enumeration familyIDs = families.keys();
         while( familyIDs.hasMoreElements() )
         {
            Family family = (Family)families.get( familyIDs.nextElement() );
            YearRange marriage = family.getMarriage();
            
            // Estimate marriage date if there actually was a marriage
            if( marriage != null )
            {
               boolean c8w = false;
               boolean c9w = false;
               boolean c8h = false;
               boolean c9h = false;
               
               // Get data for wife
               Person wife = myDB.getWife( family.getID() );
               if( wife != null )
               {
                  YearRange birthWife = wife.getBirth();
                  YearRange deathWife = wife.getDeath();
               
                  // Constraint 8 (DD >= MD)
                  c8w = applyConstraintBackward( deathWife, marriage, 0,
                           wife.getID(), DEATH, family.getID(), MARRIAGE );
                  // Constraint 9 (MD >= BD + minMarriage)
                  c9w = applyConstraintForward( 
                           marriage, birthWife, minMarriage,
                           family.getID(), MARRIAGE, wife.getID(), BIRTH );
               
                  // Check year ranges
                  checkYearRange( marriage, family.getID(), MARRIAGE );
                  checkYearRange( marriage, family.getID(), MARRIAGE );
               }

               // Get data for husband
               Person husband = myDB.getHusband( family.getID() );
               if( husband != null )
               {
                  YearRange birthHusband = husband.getBirth();
                  YearRange deathHusband = husband.getDeath();
               
                  // Constraint 8 (DD >= MD)
                  c8h = applyConstraintBackward( deathHusband, marriage, 0,
                           husband.getID(), DEATH, family.getID(), MARRIAGE );
                  // Constraint 9 (MD >= BD + minMarriage)
                  c9h = applyConstraintForward( 
                           marriage, birthHusband, minMarriage,
                           family.getID(), MARRIAGE, husband.getID(), BIRTH );
               
                  // Check year ranges
                  checkYearRange( marriage, family.getID(), MARRIAGE );
                  checkYearRange( marriage, family.getID(), MARRIAGE );
               }

               if( c8w || c9w || c8h || c9h )
               {
                  changed = true;
               }
            }
         }
      } // end while more year ranges to narrow
      return iterationCount;
   }
   
   /**
    * Applies the constraint "date1 >= date2 + variable" both forwards
    * and backwards.
    *
    * @param date1 The first date in the constraint
    * @param date2 The secons date in the constraint
    * @param variable The variable in the constraint
    * @param id1 The ID of the person/family date1 belongs to
    * @param date1Type What date date1 represents
    * @param id2 The ID of the person/family date 2 belongs to
    * @param date2Type What date date2 represents
    * @return True if a date was changed, false otherwise
    */
   private boolean applyConstraint( YearRange date1, YearRange date2, 
                     int variable, String id1, String date1Type, 
                     String id2, String date2Type )
   {
      boolean f, b;
      f = applyConstraintForward( date1, date2, variable, 
                                  id1, date1Type, id2, date2Type );
      b = applyConstraintBackward( date1, date2, variable,
                                   id1, date1Type, id2, date2Type );
      if( f || b )
      {
         return true;
      }
      return false;
   }
   
   /**
    * Applies the constraint "date1 >= date2 + variable" forwards only.
    *
    * @param date1 The first date in the constraint
    * @param date2 The secons date in the constraint
    * @param variable The variable in the constraint
    * @param id1 The ID of the person/family date1 belongs to
    * @param date1Type What date date1 represents
    * @param id2 The ID of the person/family date 2 belongs to
    * @param date2Type What date date2 represents
    * @return True if a date was changed, false otherwise
    */
   private boolean applyConstraintForward( YearRange date1, YearRange date2, 
                     int variable, String id1, String date1Type, 
                     String id2, String date2Type )
   {
      if( date1.getStart() < date2.getStart() + variable )
      {
         date1.setStart( date2.getStart() + variable );
         addLogEntry( id2, date2Type, id1, date1Type );
         return true;
      }
      return false;
   }
   
   /**
    * Applies the constraint "date1 >= date2 + variable" backwards only.
    *
    * @param date1 The first date in the constraint
    * @param date2 The secons date in the constraint
    * @param variable The variable in the constraint
    * @param id1 The ID of the person/family date1 belongs to
    * @param date1Type What date date1 represents
    * @param id2 The ID of the person/family date 2 belongs to
    * @param date2Type What date date2 represents
    * @return True if a date was changed, false otherwise
    */
   private boolean applyConstraintBackward( YearRange date1, YearRange date2, 
                     int variable, String id1, String date1Type,
                     String id2, String date2Type )
   {
      if( date2.getEnd() > date1.getEnd() - variable )
      {
         date2.setEnd( date1.getEnd() - variable );
         addLogEntry( id1, date1Type, id2, date2Type );
         return true;
      }
      return false;
   }
   
   /**
    * Checks if a given year range is negative. If it is, a 
    * NegativeYearRangeException is thrown with a message built up by
    * the other parameters.
    *
    * @param range The year range to check
    * @param id ID of the person or family this year range belongs to
    * @param date The name of the date/year range to check
    */
   private void checkYearRange( YearRange range, String id, String date ) 
   {
     if( range.isNegative() )
      {
         // Get hashtables with people and families
         Hashtable people = myDB.getPeople();
         Hashtable families = myDB.getFamilies();
         
         // Create new hashtables for people/families to be removed
         Hashtable peopleRemoved = new Hashtable();
         Hashtable familiesRemoved = new Hashtable();
         
         // Create stack and push an entry representing 'this' date on to it
         Stack stack = new Stack();
         stack.push( new DateLogEntry( id, date ) );
         Hashtable entriesDone = new Hashtable();
         
         // Loop as long as there are items on stack
         while( !stack.empty() )
         {
            // Get next entry from stack
            DateLogEntry entry = (DateLogEntry)stack.pop();
            String entryID = entry.getID();
            entriesDone.put( entryID + entry.getDate(), "" );
            
            // If date is marriage, we're dealing with a family
            if( entry.getDate().equals( MARRIAGE ) )
            {
               // Try to get the family and move it to the removed-table
               Family family = (Family)families.get( entryID );
               if( family != null )
               {
                  families.remove( entryID );
                  familiesRemoved.put( entryID, family );
               }
            }
            // Any date apart from marriage is a person
            else
            {
               // Try to get the person and move it to the removed-table
               Person person = (Person)people.get( entryID );
               if( person != null )
               {
                  people.remove( entryID );
                  peopleRemoved.put( entryID, person );
               }
            }
            // Get dates that limited 'this' date, and move them on to the stack
            Stack tempStack = (Stack)myLog.get( entryID + entry.getDate() );
            if( tempStack != null )
            {
               while( !tempStack.empty() )
               {
                  DateLogEntry newEntry = (DateLogEntry)tempStack.pop();
                  if( !entriesDone.containsKey( newEntry.getID() + 
                                                newEntry.getDate() ) )
                  {
                     stack.push( newEntry );
                  }
               }
            }
         }
         
         // If anyone where removed - add a new DB to list
         if( !peopleRemoved.isEmpty() || !familiesRemoved.isEmpty() )
         {
            GenealogyDB removed = new GenealogyDB( peopleRemoved, 
                                                   familiesRemoved );
            myErrorDBs.addElement( removed );
         }
      }
   }
   
   /**
    * Adds an entry to the log.
    *
    * @param affectorID The ID of the person/family that caused a date 
    *        to be changed
    * @param affectorDate The date in affectorID that caused the change
    * @param affectedID The ID of the person/family that had a date changed
    * @param affectedDate The date that was changed
    */
   private void addLogEntry( String affectorID, String affectorDate, 
                        String affectedID, String affectedDate )
   {
      // Try to get the log for the changed date - create it if it didn't exist
      Stack log = (Stack) myLog.get( affectedID + affectedDate );
      if( log == null )
      {
         log = new Stack();
      }

      // Create new log entry, push to log and store
      DateLogEntry entry = new DateLogEntry( affectorID, affectorDate );
      log.push( entry );
      myLog.put( affectedID + affectedDate, log );
   }
   
   /**
    * Get the log for this estimator.
    * @return the log
    */
   public Hashtable getLog()
   {
      return myLog;
   }
   
   /**
    * Get the databases that contain errors.
    * @return An enumeration of databases with errors in them
    */
   public Enumeration getErrorDBs()
   {
      return myErrorDBs.elements();
   }
}
