// File: EstimationProperties.java

// Import
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;


/**
 * This singleton class holds properties that are used in the estimation
 * process. The properies are initially read from the file properties.dat,
 * but default values are also provided in case the file is corrupted.
 * The class has methods for storing and retrieving properties, and to save
 * them to file. Only one instance of the class can exist. This instance is
 * accessed through the getInstance method.
 *
 * @author Vegard Brox
 */
public class EstimationProperties
{
   // Constants
   private static final String FILENAME = "properties.dat";

   // Static variables
   private static EstimationProperties theInstance = null;
   
   // Member variables
   private Properties myProperties = null;

   /**
    * The constructor is private to prevent multiple instances to be created.
    * It initialises the only instance with properties from file and default
    * values.
    */
   private EstimationProperties()
   {
      Properties defProperties = new Properties();
      defProperties.put( "minChildBirth", "13" );
      defProperties.put( "maxLivingAge", "120" );
      defProperties.put( "maxChildBirth", "50" );
      defProperties.put( "maxChristening", "10" );
      defProperties.put( "maxBurial", "1" );
      defProperties.put( "minMarriage", "13" );
      defProperties.put( "aboutValue", "2" );
      
      myProperties = new Properties( defProperties );
      
      try
      {
         File propertyFile = new File( System.getProperty( "user.dir" ), FILENAME );
         FileInputStream input = new FileInputStream( propertyFile );
         myProperties.load( input );
      }
      catch( IOException e )
      {
         //TODO: Is this OK?
         e.printStackTrace();
      }
   }
   
   /**
    * This method should be used to access the only instance of this class.
    * The method creates the instance if necessary, then returns it.
    *
    * @return The instance of this class
    */
   public static EstimationProperties getInstance()
   {
      if( theInstance == null )
      {
         theInstance = new EstimationProperties();
      }
      return theInstance;
   }
   
   /**
    * Retrieves a property from the property list.
    *
    * @param property The property to retrieve
    * @return The value of the property
    */
   public int getProperty( String property )
   {
      String sProperty = myProperties.getProperty( property );
      int returnValue = 0;
      try
      {
         returnValue = Integer.parseInt( sProperty );
      }
      catch( NumberFormatException e )
      {
         e.printStackTrace();
      }
      return returnValue;
   }
   
   /**
    * Stores a new value for a property in the list.
    * 
    * @param property The property to store a new value for
    * @param value The new value to store
    */
   public void setProperty( String property, int value )
   {
      myProperties.put( property, String.valueOf( value ) );
   }
   
   /**
    * Saves the current properties to file.
    */
   public void saveProperties()
   {
      try
      {
         File propertyFile = new File( System.getProperty( "user.dir" ), FILENAME );
         FileOutputStream output = new FileOutputStream( propertyFile );
         String header = "This is the property file for the GEDCOM Estimator\n#" +
                         "The values below are used in the estimation process\n#" +
                         "Don't edit this file directly unless you know what you're doing!";
         myProperties.save( output, header );
      }
      catch( IOException e )
      {
         //TODO: Is this OK?
         e.printStackTrace();
      }
   }
}
