// File: DebugOutput.java

// Import
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Stack;

/**
 * This class can print useful debug information from a given database.
 *
 * @author Vegard Brox
 */
public class DebugOutput
{
   // Private variables
   private GenealogyDB myDB;
   
   /**
    * Constructs a new debug output object for the specified database.
    * @param database The database to use
    */
   public DebugOutput( GenealogyDB database )
   {
      myDB = database;
   }

   /**
    * Prints debug information about all the people in the database.
    */
   public void printPeople()
   {
      Enumeration people = myDB.getPeople().elements();
      while( people.hasMoreElements() )
      {
         Person person = (Person)people.nextElement();
         printPerson( person );
      }
   }
   
   /**
    * Prints debug information about all the families in the database.
    */
   public void printFamilies()
   {
      Enumeration families = myDB.getFamilies().elements();
      while( families.hasMoreElements() )
      {
         Family family = (Family)families.nextElement();
         System.out.println( "Family ID: " + family.getID() );
         System.out.println( "Marriage: " + family.getMarriage() );
         System.out.println( "Husband: " + myDB.getHusband( family.getID() ) );
         System.out.println( "Wife: " + myDB.getWife( family.getID() ) );
         
         System.out.print( "Children: " );
         Enumeration children = family.getChildren();
         while( children.hasMoreElements() )
         {
            System.out.print( (String)children.nextElement() + " " );
         }
         System.out.println();
         System.out.println();
      }
   }
   
   /**
    * Prints information about a specified person.
    * @param id the ID of the person to print information about
    */
   public void printPerson( String id )
   {
      Person person = (Person)myDB.getPeople().get( id );
      printPerson( person );
   }
   
   /**
    * Prints information about a specified person.
    * @param person the person to print information about
    */
   private void printPerson( Person person )
   {
      System.out.println( "ID: " + person.getID() );
      String sex = person.isFemale() ? "female" : "male";
      System.out.println( "Sex: " + sex );
      System.out.println( "Birth: " + person.getBirth() );
      System.out.println( "Death: " + person.getDeath() );
      System.out.println( "Christening: " + person.getChristening() );
      System.out.println( "Burial: " + person.getBurial() );
      System.out.println( "Father: " + myDB.getFather( person.getID() ) );
      System.out.println( "Mother: " + myDB.getMother( person.getID() ) );
      System.out.print( "Spouse in families: " );
      Enumeration sFamilies = myDB.getSFamilies( person.getID() );
      while( sFamilies.hasMoreElements() )
      {
         System.out.print( (Family)sFamilies.nextElement() + "  " );
      }
      System.out.println();
      System.out.println();
      System.out.println();
   }
   
   /**
    * Prints the contents of a date-change log.
    * @param log a hashtable containing the log to print
    */
   public static void printLog( Hashtable log )
   {
      Enumeration keys = log.keys();
      while( keys.hasMoreElements() )
      {
         String key = (String)keys.nextElement();
         System.out.println( key + " limited by:" );
         Stack entries = (Stack)log.get( key );
         while( !entries.empty() )
         {
            DateLogEntry entry = (DateLogEntry)entries.pop();
            System.out.println( "   - " + entry );
         }
      }
   }
}

