<?php
/**
 * Defines the interface of a dependency injection container.
 *
 * @author Congyu Fan
 * @since March 16, 2017
 */

namespace Nutty\Core\IoC;

/**
 * Interface ContainerInterface
 * @package Nutty\Core\IoC
 *
 */
interface ContainerInterface
{
    /**
     * Associate a name with a Closure object which can produce an instantiated object instance.
     * null is acceptable as the return value of the given Closure object
     *
     * @param string $name
     * @param \Closure $callback
     * @return void
     */
    public function bind($name, \Closure $callback);

    /**
     * Associate a name with a Closure object which can produce an instantiated object instance.
     * null is acceptable as the return value of the given Closure object.
     * $callback will only be called once except for its return value is null.
     *
     * @param string $name
     * @param \Closure $callback
     * @return void
     */
    public function bindSingleton($name, \Closure $callback);

    /**
     * Associate a name with an object instance.
     * null is NOT acceptable.
     *
     * @param string $name
     * @param object $instance
     * @return void
     */
    public function bindInstance($name, $instance);

    /**
     * Associate an interface name with an implementation.
     *
     * @param string $interfaceName
     * @param string $implName
     * @return bool false if $implName is not an implementation of $interfaceName
     */
    public function bindInterface($interfaceName, $implName);

    /**
     * Produce an object instance by a given name.
     *
     * @param string $name Can be either a name which has been bound with the container before or
     *                      a name of a class or an interface. If it is an interface name, bindInterface()
     *                      should be called in advance.
     *
     * @param array $args If a Closure object pr the constructor of a class is called,
     *                    $args will be passed to it as arguments.
     *                    $args should be in "name" => "value" pairs.
     *
     * @return bool|null|object
     *                    If a Closure object returns null, null will be returned.
     *                    If an object is produced successfully, an object instance will be returned.
     *                    Otherwise false will be returned.
     *
     */
    public function make($name, array $args = []);

    /**
     * Try to obtain an object by calling make() method and invoke a method of this object.
     *
     * @param string $name A string which will be passed to make() method.
     * @param string $methodName The name of the method which will be invoked.
     * @param array $methodArgs The arguments which will be passed to the method.
     *                          Should be in "name" => "value" pairs.
     *
     * @param array $args The arguments which will be passed to the make() method.
     *                          Should be in "name" => "value" pairs.
     *
     * @return mixed The return value of the invoked method.
     *
     * @throws \BadMethodCallException if the methodName is invalid.
     *         \RuntimeException if the make() cannot produce an object instance,
     */
    public function makeAndExecute($name, $methodName, array $methodArgs = [], array $args = []);
}