<?php
/**
 * Provide advanced functionality of a dependency injection container. For example, the ability
 * of resolving dependencies.
 * Inspired by Laravel framework.
 *
 * @author Congyu Fan
 * @since March 16, 2017
 */

namespace Nutty\Core\IoC;

/**
 * Class Container
 * @package Nutty\Core\IoC
 */
class Container extends AbstractContainer
{
    private function __construct() { }

    /**
     * @inheritDoc
     */
    protected final function resolveReflectionFunctionParam(\ReflectionFunctionAbstract $function, array $args)
    {
        //get the parameter list of the reflection function
        $paramList = $function->getParameters();

        // To store the return value
        $argList = [];

        //for each parameter
        foreach ($paramList as $param) {
            //Get the name of the parameter
            $paraName = $param->getName();

            //Check if the argument is provided
            if (array_key_exists($paraName, $args)) {
                array_push($argList, $args[$paraName]);
                continue;
            }

            //Try to get the hinted type of the parameter
            $paraClass = $param->getClass();

            //if type hinted
            if ($paraClass) {
                $obj = $this->make($paraClass->getName());

                if ($obj !== false) {
                    array_push($argList, $obj);
                    continue;
                }
            }

            //Try to use the default value.
            array_push($argList, $param->getDefaultValue());
        }

        return $argList;
    }

    /**
     * Try to instantiate an object by a given name and automatically resolve its constructor dependency.
     *
     * @param string $name class name
     * @param array $args Arguments provided by the user.
     * @return bool|object false if failed to resolve the dependency.
     */
    private function resolveDependencyByClassName($name, $args)
    {
        if (!class_exists($name, true)) {
            return false;
        }

        //get the reflection class.
        $reflectionClass = new \ReflectionClass($name);

        //get the arguments for the constructor
        $argList = $this->resolveReflectionFunctionParam($reflectionClass->getConstructor(), $args);

        if (is_null($argList)) {
            return false;
        }

        //instantiate it
        return $reflectionClass->newInstanceArgs($argList);
    }

    /**
     * Get the unique instance of Container.
     *
     * @return Container
     */
    public static function getInstance()
    {
        static $selfInstance = null;

        //if not instantiated
        if (!$selfInstance) {
            //create a new instance
            $selfInstance = new self();
        }

        return $selfInstance;
    }


    /**
     * @inheritDoc
     */
    public function make($name, array $args = [])
    {
        $obj = parent::make($name, $args);

        if ($obj !== false) {
            return $obj;
        }

        //if failed, try to resolve the dependency automatically.
        return $this->resolveDependencyByClassName($name, $args);
    }

    /**
     * @inheritDoc
     */
    public function makeAndExecute($name, $methodName, array $methodArgs = [], array $args = []) {
        $obj = $this->make($name, $args);

        //if got an instance successfully, try to invoke the method.
        if (is_object($obj)) {
            $reflection = new \ReflectionClass($obj);
            $argList = $this->resolveReflectionFunctionParam($reflection->getMethod($methodName), $methodArgs);

            if ($argList === null) {
                throw new \RuntimeException("Arguments of method $methodName cannot be resolved automatically");
            }

            return call_user_func_array([$obj, $methodName], $argList);
        }

        throw new \RuntimeException("object $name cannot be created");
    }
}