<?php
/**
 * Provide basic functionality of a dependency injection container.
 * Inspired by Laravel framework.
 *
 * @author Congyu Fan
 * @since March 16, 2017
 */

namespace Nutty\Core\IoC;

/**
 * Class AbstractContainer
 * @package Nutty\Core\IoC
 */
abstract class AbstractContainer implements ContainerInterface
{
    /** @var array $bindings to store binding settings for callable objects. */
    private $bindings = [];

    /** @var array $instances To store the instantiated objects for singleton mode.*/
    private $instances = [];

    /** @var array $interfaceAliases Key: InterfaceName, Value: Implementation*/
    private $interfaceAliases = [];

    /**
     * Try to resolve the dependency of a reflection function.
     *
     * @param \ReflectionFunctionAbstract $function A reflection function
     * @param array $args an argument list provided in "arg name" => "value" pairs
     * @return array An array of resolved arguments
     */
    protected abstract function resolveReflectionFunctionParam(\ReflectionFunctionAbstract $function, array $args);

    /**
     * Associate a name with a callable object.
     *
     * @param string $name
     * @param \Closure $callback
     * @param bool $singleton true if singleton mode
     *
     * @return void
     */
    private function bindCallable($name, \Closure $callback, $singleton = false)
    {
        $this->bindings[$name] = [$callback, $singleton];
    }

    /**
     * @inheritDoc
     */
    public function bind($name, \Closure $callback)
    {
        $this->bindCallable($name, $callback, false);
    }

    /**
     * @inheritDoc
     */
    public function bindSingleton($name, \Closure $callback)
    {
        $this->bindCallable($name, $callback, true);
    }

    /**
     * @inheritDoc
     */
    public function bindInstance($name, $instance)
    {
        $this->instances[$name] = $instance;
    }

    /**
     * @inheritDoc
     */
    public function make($name, array $args = [])
    {
        //if there is an existing instance
        if (array_key_exists($name, $this->instances)) {
            return $this->instances[$name];
        }

        //If the name was associated with a callable object
        if (array_key_exists($name, $this->bindings)) {
            $binding = $this->bindings[$name];

            /*//first argument should be the container
            array_unshift($args, $this);

            //invoke the callable object.
            $obj = call_user_func_array($binding[0], $args);*/

            $refClosure = new \ReflectionFunction($binding[0]);

            $args = $this->resolveReflectionFunctionParam($refClosure, $args);
            if (is_null($args)) {
                return false;
            }

            $obj = call_user_func_array($binding[0], $args);

            //if singleton mode
            if ($binding[1]) {
                //if the result is an instance, bind it with the container.
                if (is_object($obj)) {
                    $this->bindInstance($name, $obj);
                }
            }

            return $obj;
        }

        //If an interface is bound with an implementation
        if (array_key_exists($name, $this->interfaceAliases)) {
            return $this->make($this->interfaceAliases[$name]);
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function bindInterface($interfaceName, $implName)
    {
        $interfaces = class_implements($implName, true);

        //check if $implName is an implementation of $interfaceName
        if (!in_array($interfaceName, $interfaces)) {
            return false;
        }

        //store interface binding.
        $this->interfaceAliases[$interfaceName] = $implName;
        return true;
    }
}