<?php
/**
 * To bind an alias with a class. Inspired by Laravel framework.
 *
 * @author Congyu Fan
 * @since March 16, 2017
 */

namespace Nutty\Core\Facade;

/**
 * Class ClassAlias
 * @package Nutty\Core\Facade
 */
final class ClassAlias
{
    /**
     * @var array $aliases Used to store aliases
     */
    private $aliases = [];

    /**
     * ClassAlias constructor.
     */
    private function __construct()
    {
        //Read in the configuration
        $this->aliases = $providerList = ntGetConfigFile(NUTTY_CONFIG_DIR.'/app.php')['aliases'];

        //register an autoload function which will be called when the user try to reference an undefined class.
        spl_autoload_register([$this, '__loadHandler'], true, false);
    }

    /**
     * autoload handler
     *
     * @param string $className
     * @return void
     */
    private function __loadHandler($className)
    {
        //Find the position of the last \
        $pos = strrpos($className, '\\');

        //if not found, set $pos to -1
        if ($pos === false) {
            $pos = -1;
        }

        //Treat the part of $className which is after the last \ as the alias.
        $alias = substr($className, $pos + 1);


        //if the alias is registered
        if (array_key_exists($alias, $this->aliases)) {
            //Try to bind with the target class. If fail, throw an exception.
            if (class_alias($this->aliases[$alias], $className) === false) {
                throw new \InvalidArgumentException("[Nutty\\ClassAlias] 1#: Failed to give $className an alias");
            }
        }
    }

    /**
     * Get the unique instance of this class.
     *
     * @return ClassAlias
     */
    public static function getInstance()
    {
        static $myInstance;

        if (!$myInstance) {
            $myInstance = new self;
        }

        return $myInstance;
    }
}