<?php
/**
 * To implement a simple Facade. Inspired by Laravel framework.
 *
 * @author Congyu Fan
 * @since March 16, 2017
 */

namespace Nutty\Core\Facade;

use Nutty\Core\IoC\Container;

/**
 * Class AbstractFacade
 * @package Nutty\Core\Facade
 */
abstract class AbstractFacade
{

    private function __construct() { }

    /**
     * Get the name used to register with the container.
     * @return string
     */
    abstract protected function getRegisterName();

    /**
     * To implement the singleton mode of AbstractFacade
     *
     * @return object
     */
    final private static function getInstance()
    {
        //get called class name.
        $className = get_called_class();

        static $myInstance = [];

        //if the called class has not been initialized
        if (!array_key_exists($className, $myInstance)) {
            $myInstance[$className] = new $className;
        }

        return $myInstance[$className];
    }

    /**
     * Use magic method to implement "pseudo static call"
     *
     * @param string $name
     * @param array $arguments
     * @return mixed
     */
    final public static function __callStatic($name , array $arguments)
    {
        //get the facade instance.
        $facade = self::getInstance();
        //get the object from the container.
        $obj = Container::getInstance()->make($facade->getRegisterName());

        //check if this is an valid method.
        if (!method_exists($obj, $name)) {
            throw new \BadMethodCallException('[Nutty\Facades] 1#: Trying to call an invalid method');
        }

        //call the method.
        return call_user_func_array([$obj, $name], $arguments);
    }
}