<?php
/**
 * Service provider for wrapper classes of models for staff apis.
 *
 * @author Congyu Fan
 */

namespace App\Providers;


use App\Models\StaffDefaults\StaffHuntWrapper;
use App\Models\StaffDefaults\StaffLocationWrapper;
use App\Models\StaffDefaults\StaffLocTemplateWrapper;
use App\Models\StaffDefaults\StaffQuizOptionWrapper;
use App\Models\StaffDefaults\StaffQuizQuestionWrapper;
use App\Models\StaffDefaults\StaffTeamWrapper;
use Nutty\Core\IoC\AbstractServiceProvider;
use Nutty\Models\Base\LocationQuery;
use Nutty\Models\Base\LocationTemplateQuery;
use Nutty\Models\HuntQuery;
use Nutty\Models\QuizOptionQuery;
use Nutty\Models\QuizQuestionQuery;
use Nutty\Models\TeamQuery;
use Respect\Validation\Validator;

/**
 * Class StaffModelsProvider
 * @package App\Providers
 */
class StaffModelsProvider extends AbstractServiceProvider
{
    /**
     * @inheritdoc
     */
    public function register()
    {
        // For StaffHuntWrapper
        $this->container->bindSingleton(
            StaffHuntWrapper::class,
            function() {
                if (!Auth::isAuthenticated()) {
                    return null;
                }

                $huntID = ntGetRequestInput('hunt_id');

                //If invalid hunt id
                if (!Validator::intVal()->validate($huntID)) {
                    return null;
                }

                //Get hunt
                $hunt = HuntQuery::create()
                    ->filterByHuntID($huntID)
                    ->filterByCreatorID(Auth::getIdentification())  //Should check the creator.
                    ->findOne();

                return $hunt ? new StaffHuntWrapper($hunt) : null;
            }
        );

        // For StaffLocationWrapper
        $this->container->bindSingleton(
            StaffLocationWrapper::class,
            function(StaffHuntWrapper $huntWrapper = null) {
                //Try to get the hunt first. If not found, return null.
                if (!$huntWrapper) {
                    return null;
                }

                $locationOrder = ntGetRequestInput('order');
                //IF invalid order number
                if (!Validator::intVal()->validate($locationOrder)) {
                    return null;
                }

                //Try to find the location.
                $location = LocationQuery::create()
                    ->filterByHuntID($huntWrapper->getWrapped()->getHuntID())
                    ->filterByLocationOrder($locationOrder)
                    ->findOne();

                return $location ? new StaffLocationWrapper($location) : null;
            }
        );

        // For StaffLocTemplateWrapper
        $this->container->bindSingleton(
            StaffLocTemplateWrapper::class,
            function() {
                //If not authenticated.
                if (!Auth::isAuthenticated()) {
                    return null;
                }

                $templateID = ntGetRequestInput('template_id');

                //Validate template id
                if (!Validator::intVal()->validate($templateID)) {
                    return null;
                }

                //Find location template.
                $template = LocationTemplateQuery::create()
                    ->filterByCreatorID(Auth::getIdentification())
                    ->filterByLocationID($templateID)
                    ->findOne();

                return $template ? new StaffLocTemplateWrapper($template) : null;
            }
        );

        // For StaffQuizQuestionWrapper
        $this->container->bindSingleton(
            StaffQuizQuestionWrapper::class,
            function () {
                //If not authenticated.
                if (!Auth::isAuthenticated()) {
                    return null;
                }

                $questionID = ntGetRequestInput('question_id');

                //Validate question id
                if (!Validator::intVal()->validate($questionID)) {
                    return null;
                }


                //Try to find the question
                $question = QuizQuestionQuery::create()
                    ->filterByQuestionID($questionID)
                    ->joinWithHunt()
                    ->where('Hunt.CreatorID = ?', Auth::getIdentification())
                    ->findOne();

                //If not found
                if (!$question) {
                    return null;
                }

                //Check the creator.
                if ($question->getHunt()->getCreatorID() != Auth::getIdentification()) {
                    return null;
                }

                return new StaffQuizQuestionWrapper($question);
            }
        );

        //For StaffQuizOptionWrapper
        $this->container->bindSingleton(
            StaffQuizOptionWrapper::class,
            function (StaffQuizQuestionWrapper $question = null) {
                //Try to get the question first. If not found, return null.
                if (!$question) {
                    return null;
                }

                $order = ntGetRequestInput('order');

                //Validate order number.
                if (!Validator::intVal()->validate($order)) {
                    return null;
                }

                // Find the option.
                $option = QuizOptionQuery::create()
                    ->filterByQuestionID($question->getWrapped()->getQuestionID())
                    ->filterByQuizOrder($order)
                    ->findOne();


                return $option ? (new StaffQuizOptionWrapper($option)) : null;
            }
        );

        //For StaffTeamWrapper
        $this->container->bindSingleton(
            StaffTeamWrapper::class,
            function() {
                //Find team
                $team = TeamQuery::create()
                    ->joinWithHunt()
                    ->where('Hunt.CreatorID = ?', Auth::getIdentification())
                    ->filterByTeamID(ntGetRequestInput('team_id'))
                    ->findOne();

                if (!$team) {
                    return null;
                }

                return new StaffTeamWrapper($team);
            }
        );
    }
}