<?php
/**
 * A wrapper class for Team model.
 *
 * @author Congyu Fan
 * @since Mar 27, 2017
 */

namespace App\Models;


use LucDeBrouwer\Distance\Distance;
use Nutty\Models\Base\Location;
use Nutty\Models\Base\LocationQuery;
use Nutty\Models\Base\Team;
use Nutty\Models\QuizQuestionQuery;
use Propel\Runtime\Connection\ConnectionInterface;
use Nutty\Models\Base\QuizQuestion;


/**
 * Class TeamWrapper
 * @package App\Models
 */
class TeamWrapper extends AbstractWrapper
{

    private $gameRecord = null;
    private $locations = null;
    private $questions = null;

    /**
     * @param Team $team A Team model
     */
    public function __construct(Team $team)
    {
        $this->wrapped = $team;
    }

    /**
     * @return Team
     */
    public function getWrapped()
    {
        return $this->wrapped;
    }

    /**
     * Get the game record of the wrapped team
     *
     * @param ConnectionInterface|null $con
     * @return null|\Nutty\Models\GameRecord
     */
    public function getGameRecord(ConnectionInterface $con = null)
    {
        //If game record not stored
        if (!$this->gameRecord) {
            //Retrieve from the DB and store it
            $this->gameRecord = $this->getWrapped()->getGameRecord($con);
        }

        return $this->gameRecord;
    }

    /**
     * Get the location list of the hunt which the current team is in.
     *
     * @return int The number of locations.
     */
    protected function retrieveLocations()
    {
        //If locations not stored, retrieve from database
        if (!$this->locations) {
            $this->locations = LocationQuery::create()
                ->filterByHuntID($this->getWrapped()->getHuntID())
                ->orderByLocationOrder()
                ->find();
        }

        //Return the number of locations.
        return $this->locations->count();
    }

    /**
     * Get the question list of the hunt which the current team is in.
     *
     * @return int The number of questions
     */
    private function retrieveQuestions()
    {
        //If questions not stored, retrieve from database
        if (!$this->questions) {
            $this->questions = QuizQuestionQuery::create()
                ->filterByHuntID($this->wrapped->getHuntID())
                ->orderByQuestionID()
                ->find();
        }

        return $this->questions->count();
    }

    /**
     * Get a location with a location number.
     * Note: The "order" here is not the "order" stored in the database
     * but a sequence number.
     *
     * @param int $orderNum
     * @return Location|null
     */
    protected function getLocationByOrder($orderNum = 0)
    {
        //Get location list
        $count = $this->retrieveLocations();

        //If there's no location
        if ($count <= 0) {
            return null;
        }

        //If invalid order provided.
        if ($orderNum < 0 || $orderNum > $count - 1) {
            return null;
        }

        /**
         * If provided order number implies the first or the last location
         */
        if ($orderNum == 0 || $orderNum == $count - 1) {
            return $this->locations->offsetGet($orderNum);
        }

        /**
         * If implies a location which is between the first and the last, calculate to get
         * that location based on the specified offset number for the team.
         */
        $offset = $this->getWrapped()->getLocOffset();
        $index = (($offset + $orderNum - 1) % ($count - 2)) + 1;
        return $this->locations->offsetGet($index);
    }

    /**
     * Get the question by a given question order
     *
     * @param int $orderNum
     * @return null
     */
    protected function getQuestionByOrder($orderNum = 0)
    {
        //Get the question list
        $count = $this->retrieveQuestions();

        //If no question
        if ($count <= 0) {
            return null;
        }

        //If invalid order number
        if ($orderNum < 0 || $orderNum > $count - 1) {
            return null;
        }

        return $this->questions->offsetGet($orderNum);
    }

    /**
     * Get the location that the team is looking for
     *
     * @return Location|null
     */
    protected function getCurrentLoc()
    {
        return $this->getLocationByOrder($this->getGameRecord()->getCluesFound());
    }

    /**
     * Get the question that the team is answering.
     *
     * @return QuizQuestion|null
     */
    public function getCurrentQuestion()
    {
        return $this->getQuestionByOrder($this->getGameRecord()->getQuestion());
    }


    /**
     * Get the number of questions
     *
     * @return int The number of questions
     */
    public function getTotalQuestionNum()
    {
        return $this->retrieveQuestions();
    }

    /**
     * Get next location
     *
     * @return null|Location
     */
    protected function getNextLoc()
    {
        return $this->getLocationByOrder($this->getGameRecord()->getCluesFound() + 1);
    }

    /**
     * Calculate the total distance for the team based on the offset number assigned
     * to this team.
     *
     * @return int
     */
    public function calTotalDist()
    {
        $count = $this->retrieveLocations();
        $i = 1;

        /** @var int $totalDist to store the total distance.*/
        $totalDist = 0;
        $distance = App::make(Distance::class);

        // For each pair of locations we need to calculate the
        // Distance between them and sum the results together.
        while ($i < $count) {
            $loc1 = $this->getLocationByOrder($i - 1);
            $loc2 = $this->getLocationByOrder($i);
            $totalDist += $distance->between(
                $loc1->getLatitude(),
                $loc1->getLongitude(),
                $loc2->getLatitude(),
                $loc2->getLongitude()
            );

            ++ $i;
        }

        return $totalDist;
    }

    /**
     * Calculate the time for the next location to display the second hint.
     *
     * @return int
     */
    protected function calRevealTime()
    {
        $totalDist = $this->wrapped->getTotalDist();

        if (!$totalDist) {
            return 0;
        }

        $current = $this->getCurrentLoc();
        $next = $this->getNextLoc();
        $distance = App::make(Distance::class);

        // If cannot get the current or the next location.
        if (!($current && $next)) {
            return 0;
        }

        $duration = $this->getWrapped()->getHunt()->getDuration();

        //Calculate the distance from the current location to the next.
        $nextDist = $distance->between(
            $current->getLatitude(),
            $current->getLongitude(),
            $next->getLatitude(),
            $next->getLongitude()
        );

        //calculate the reveal time.
        return (int)(($nextDist / $totalDist) * $duration);
    }

    /**
     * Check if all locations are found by the team
     *
     * @return bool
     */
    public function foundAll()
    {
        //If location list was not retrieved from the DB
        if (!$this->locations) {
            //Get the location number
            $count = LocationQuery::create()
                ->filterByHuntID($this->getWrapped()->getHuntID())
                ->count();
        } else {
            $count = $this->locations->count();
        }

        //Get the number of locations found by the team.
        $cluesFound = $this->getGameRecord()->getCluesFound();

        return $cluesFound >= $count;
    }

    /**
     * Check if all questions are answered.
     *
     * @return bool
     */
    public function answeredAll()
    {
        $count = $this->getTotalQuestionNum();
        $answered = $this->getGameRecord()->getQuestion();
        return $answered >= $count;
    }

    /**
     * Get the remaining time until displaying the second hint for the current location.
     *
     * @return int
     */
    public function getClueTimeLeft()
    {
        $gameRecord = $this->getGameRecord();
        /** @var int $revealTime The time to display the second hint for the current location*/
        $revealTime = $gameRecord->getRevealTime();
        /** @var int $timeFirstQuery The first time the user get the first hint*/
        $timeFirstQuery = $gameRecord->getTimeFirstGetLoc();

        // If the team has not yet obtained the current location
        if ($timeFirstQuery <= 0) {
            return $revealTime;
        }

        $timeLeft = $revealTime - (time() - $timeFirstQuery);
        return $timeLeft > 0 ? $timeLeft : 0;
    }
}
