<?php
/**
 * Wrapper class for Team model for Team APIs
 *
 * @author Congyu Fan
 */

namespace App\Models\TeamDefaults;


use App\Misc\Exception\RequestErrorException;
use App\Misc\Validation\Validation;
use App\Models\TeamWrapper;
use App\Misc\API\KeyGenerator;
use Azi\Input;
use Kayex\HttpCodes;
use LucDeBrouwer\Distance\Distance;
use Nutty\Models\Base\Team;
use Nutty\Models\TeamQuery;
use Propel\Runtime\Connection\ConnectionInterface;

/**
 * Class TeamTeamWrapper
 * @package App\Models\TeamDefaults
 */
class TeamTeamWrapper extends TeamWrapper
{
    const CURR_LOC_STATUS_NORMAL = 0;   // If the location information is returned as normal
    const CURR_LOC_STATUS_SPECIAL_LOC = 1;   // If this is a special location
    const CURR_LOC_STATUS_FINISHED = 2; // If the player has finished the hunt.

    const REACH_LOC_STATUS_INVALID_RANGE = 3;
    const REACH_LOC_STATUS_INVALID_CODE = 2;    // If the location identification submitted doesn't match with the current.
    const REACH_LOC_STATUS_FINISHED = 1;    //If the user has finished the hunt.
    const REACH_LOC_STATUS_NORMAL = 0;      //If the clues counter is updated as normal

    const QUIZ_ANSWERED_ALL = 1;
    const QUIZ_NORMAL = 0;

    /**
     * TeamTeamWrapper constructor.
     * @param Team $team
     */
    public function __construct(Team $team)
    {
        parent::__construct($team);
    }

    /**
     * Get the location that the team is looking for,
     * If this is the first time the team request for this location,
     * The timestamp in game record will be updated.
     *
     * @param ConnectionInterface|null $con
     * @return null|\Nutty\Models\Base\Location
     */
    protected function getCurrentLocAndUpdateTime(ConnectionInterface $con = null)
    {
        //Get the current location.
        $location = parent::getCurrentLoc();
        //if fail, return null.
        if (!$location) {
            return null;
        }

        $gameRecord = $this->getGameRecord($con);

        //If this is the first time the team request for this location
        //Update game record.
        if (!$gameRecord->getTimeFirstGetLoc()) {
            $gameRecord->setTimeFirstGetLoc(time());
            $gameRecord->save($con);
        }

        return $location;
    }

    /**
     * For a team to move to the next location
     *
     * @param Distance $distanceCal Distance calculator
     * @param ConnectionInterface|null $con
     * @return mixed
     * @throws RequestErrorException
     */
    public function moveToNextLoc(Distance $distanceCal, ConnectionInterface $con = null)
    {
        $locationKey = ntGetRequestInput('key');
        $latitude = ntGetRequestInput('latitude');
        $longitude = ntGetRequestInput('longitude');

        if (!Validation::coordinates($latitude, $longitude)) {
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, 'Illegal coordinates');
        }

        $latitude = doubleval($latitude);
        $longitude = doubleval($longitude);

        if ($this->foundAll()) {
            $ret['status'] = self::REACH_LOC_STATUS_FINISHED;
            return $ret;
        }

        //Get the current location
        $location = $this->getCurrentLoc();

        $userDistance = $distanceCal->between(
            $location->getLatitude(),
            $location->getLongitude(),
            $latitude,
            $longitude
        );

        //Check if the user is close enough to the location
        if ($userDistance > ntLoadGlobalConfig('max_scan_range')) {
            $ret['status'] = self::REACH_LOC_STATUS_INVALID_RANGE;
            return $ret;
        }

        //check the location identification
        if (($locationKey != KeyGenerator::locationKey($location))) {
            $ret['status'] = self::REACH_LOC_STATUS_INVALID_CODE;
            return $ret;
        }

        $gameRecord = $this->getGameRecord($con);

        $revealTime = $gameRecord->getRevealTime();
        $timeLeft = $this->getClueTimeLeft();;

        //Calculate mark
        if ($timeLeft > $revealTime) {
            throw new RequestErrorException(HttpCodes::HTTP_EXPECTATION_FAILED, 'Unexpected time spent.');
        } else {
            if (!$revealTime) {
                $mark = 0;
            } else {
                $mark = ($timeLeft / $revealTime) * ntLoadGlobalConfig('max_mark_per_location');
            }
        }

        // Add some extra marks so that no team can get 0
        $mark += ntLoadGlobalConfig('base_mark_per_location');
        $ret['mark'] = $mark;

        $gameRecord->setRank($gameRecord->getRank() + $mark);
        //Set the reveal time.
        $gameRecord->setRevealTime($this->calRevealTime());

        //Increment the counter. ***MUST*** be executed AFTER calculating the reveal time.
        $gameRecord->setCluesFound($gameRecord->getCluesFound() + 1);

        //Reset the timestamp.
        $gameRecord->setTimeFirstGetLoc(0);
        $gameRecord->save($con);

        if ($this->foundAll()) {
            $ret['status'] = self::REACH_LOC_STATUS_FINISHED;
        } else {
            $ret['status'] = self::REACH_LOC_STATUS_NORMAL;
        }

        $ret['name'] = $location->getName();
        return $ret;
    }

    /**
     * Get the location that the team is currently looking for,
     *
     * @param TeamHuntWrapper $huntWrapper
     * @param ConnectionInterface|null $con
     * @return array
     */
    public function retrieveLoc(TeamHuntWrapper $huntWrapper,ConnectionInterface $con = null)
    {
        /** @var array $ret An array used to store the information that will be sent to the user.*/
        $ret = [];

        //if the team has found every clue, then the array with all the information that the user needs will be sent.
        //the status of the hunt of a team will be set as finished.
        if ($this->foundAll()) {
            $ret['status'] = self::CURR_LOC_STATUS_FINISHED;
            return $ret;
        }

        $gameRecord = $this->getGameRecord();
        // If the number of clues solved by a team is larger than 0,
        // then this is not a special location.
        if ($gameRecord->getCluesFound() > 0) {
            $ret['status'] = self::CURR_LOC_STATUS_NORMAL;
        } else {
            $ret['status'] = self::CURR_LOC_STATUS_SPECIAL_LOC;
        }

        $location = $this->getCurrentLocAndUpdateTime($con);

        //Sets the name field of the array to be the fetched name.
        $ret['name'] = $location->getName();
        //Fetches the clue and sets it to the variable $clue.
        $clue = $location->getClue();
        $ret['clue'] = $clue ? $clue : '';
        //Sets the 'time_left' field to the amount of time left of a clue for a team.
        $ret['time_left'] = $this->getClueTimeLeft();
        //Sets the time_spent field to be determined by the starting location of a team.
        $ret['time_spent'] = time() - $gameRecord->getTimeFirstGetLoc();

        // If no time remains for the current location, attach the coordinates.
        if (!$ret['time_left'] && !$huntWrapper->getWrapped()->getDifficulty()) {
            $ret['latitude'] = $location->getLatitude();
            $ret['longitude'] = $location->getLongitude();
        }

        return $ret;
    }

    /**
     * Change the team name
     *
     * @param ConnectionInterface|null $con
     * @throws RequestErrorException
     */
    public function updateName(ConnectionInterface $con = null)
    {
        $newName = ntGetRequestInput('name');

        // If no valid name is submitted.
        if (!Validation::teamName($newName)) {
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, 'Invalid Name. ');
        }

        // Check bad words
        if (Validation::badWords($newName)) {
            $errMsg = 'Team Name '.ntGetStringLiteral('CONTAIN_BAD_WORD');
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, $errMsg);
        }

        //Check if there exists an identical team name,
        $exist = TeamQuery::create()
            ->filterByHuntID($this->getWrapped()->getHuntID())
            ->filterByTeamName($newName)
            ->exists($con);

        if ($exist) {
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, 'Duplicate name');
        }

        $this->getWrapped()->setTeamName($newName)->save($con);
    }

    /**
     * Get the quiz question that the team is working on
     *
     * @return array
     * @throws RequestErrorException
     */
    public function retrieveQuestion()
    {
        // If the team has not found all locations
        if (!$this->foundAll()) {
            throw new RequestErrorException(HttpCodes::HTTP_NOT_FOUND, 'You have not found all locations yet.');
        }

        // get the number of answered questions
        $ret['answered'] = $this->getGameRecord()->getQuestion();
        //Get total question number
        $ret['total'] = $this->getTotalQuestionNum();

        //Get the current question
        $question = $this->getCurrentQuestion();
        //If cannot get the current question
        if (!$question) {
            //If all questions are answered.
            if ($this->answeredAll()) {
                return $ret;
            } else {
                throw new RequestErrorException(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, 'Failed to retrieve the current question');
            }
        }

        $ret['question'] = $question->getQuestion();
        $ret['options'] = [];

        $options = $question->getQuizOptions();

        foreach ($options as $option) {
            array_push($ret['options'], [
                'order' => $option->getQuizOrder(),
                'option' => $option->getQuizOption()
            ]);
        }

        return $ret;
    }

    /**
     * For a team to submit a question answer.
     *
     * @param ConnectionInterface|null $con
     * @return array
     * @throws RequestErrorException
     */
    public function updateQuestionCounter(ConnectionInterface $con = null)
    {
        $question = $this->getCurrentQuestion();

        // If cannot retrieve the current question
        if (!$question) {
            if ($this->answeredAll()) { //If all questions are answered.
                $ret['status'] = self::QUIZ_ANSWERED_ALL;
                return $ret;
            } else {
                throw new RequestErrorException(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, 'Failed to retrieve the current question');
            }
        }

        /** @var string $answerOrder The order number of the answer submitted by the user*/
        $answerOrder = ntGetRequestInput('order');
        $gameRecord = $this->getGameRecord();

        /** @var int $rightANS The order number of the correct answer.*/
        $rightANS = $question->getRightANS();

        if ($answerOrder == $rightANS) {
            //Correct
            $gameRecord->setRank($gameRecord->getRank() + ntLoadGlobalConfig('max_mark_per_question'));
        }

        $ret['correct'] = $rightANS;

        //Increment the counter in the game record.
        $gameRecord->setQuestion(
            $gameRecord->getQuestion() + 1
        );

        $gameRecord->save($con);

        // If answered all.
        if ($this->answeredAll()) {
            $ret['status'] = self::QUIZ_ANSWERED_ALL;
        } else {
            $ret['status'] = self::QUIZ_NORMAL;
        }

        return $ret;
    }
}