<?php
/**
 * Wrapper class for Hunt model for team APIs
 *
 * @author Congyu Fan
 */

namespace App\Models\TeamDefaults;


use App\Misc\Exception\RequestErrorException;
use App\Models\HuntWrapper;
use Kayex\HttpCodes;
use Nutty\Models\Base\Hunt;
use Nutty\Models\TeamQuery;
use Propel\Runtime\ActiveQuery\Criteria;


/**
 * Class TeamHuntWrapper
 * @package App\Models\TeamDefaults
 */
class TeamHuntWrapper extends HuntWrapper
{
    /**
     * TeamHuntWrapper constructor.
     * @param Hunt $hunt
     */
    public function __construct(Hunt $hunt)
    {
        parent::__construct($hunt);
    }

    /**
     * Get the live scoreboard
     *
     * @return array
     * @throws RequestErrorException
     */
    public function getLiveScoreBoard()
    {
        //if the hunt is inactive.
        if (!$this->isActive()) {
            throw new RequestErrorException(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_INACTIVE'));
        }

        $teams = TeamQuery::create()
            ->joinWithGameRecord()
            ->filterByHuntID($this->getWrapped()->getHuntID())
            ->orderBy('GameRecord.CluesFound', Criteria::DESC)
            ->find();

        //$ret will be the array which holds all the data about the teams.
        $ret = [];
        //Iterate through all the teams and insert the team name and the number of clues they found into the array.
        foreach ($teams as $team) {
            $teamName = $team->getTeamName();
            array_push(
                $ret,
                [
                    'team_name' => $teamName ? $teamName : 'unnamed',
                    'clues_found' => $team->getGameRecord()->getCluesFound(),
                    'self' => (int)($team->getTeamID() == Auth::getIdentification())
                ]
            );
        }

        return $ret;
    }

    /**
     * Get the final scoreboard.
     *
     * @return array
     * @throws RequestErrorException
     */
    public function getFinalScore() {
        if ($this->getTimeLeft() === -1) {
            throw new RequestErrorException(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_START'));
        }

        $teams = TeamQuery::create()
            ->joinWithGameRecord()
            ->filterByHuntID($this->getWrapped()->getHuntID())
            ->orderBy('GameRecord.Rank', Criteria::DESC)
            ->find();

        //$ret will be the array which holds all the data about the teams.
        $ret = [];
        //Iterate through all the teams and insert the team name and the number of clues they found into the array.
        foreach ($teams as $team) {
            $teamName = $team->getTeamName();
            array_push(
                $ret,
                [
                    'team_name' => $teamName ? $teamName : 'unnamed',
                    'mark' => $team->getGameRecord()->getRank(),
                    'self' => (int)($team->getTeamID() == Auth::getIdentification())
                ]
            );
        }

        return $ret;
    }
}