<?php
/**
 * Wrapper class for QuizQuestion Model for Staff APIs
 *
 * @author Congyu Fan
 */

namespace App\Models\StaffDefaults;


use App\Misc\Exception\RequestErrorException;
use App\Models\QuizQuestionWrapper;
use Kayex\HttpCodes;
use Nutty\Models\Base\QuizQuestion;
use Nutty\Models\Base\QuizOption;
use Nutty\Models\Base\QuizOptionQuery;
use Propel\Runtime\ActiveQuery\Criteria;
use Propel\Runtime\Connection\ConnectionInterface;


/**
 * Class StaffQuizQuestionWrapper
 * @package App\Models\StaffDefaults
 */
class StaffQuizQuestionWrapper extends QuizQuestionWrapper
{
    /**
     * StaffQuizQuestionWrapper constructor.
     * @param QuizQuestion $question
     */
    public function __construct(QuizQuestion $question)
    {
        parent::__construct($question);
    }

    /**
     * For adding a new option
     *
     * @param ConnectionInterface $con
     * @return int The order number of the newly created option
     * @throws RequestErrorException
     */
    public function addOption(ConnectionInterface $con)
    {
        if ((new StaffHuntWrapper($this->getWrapped()->getHunt()))->isActive()) {
            throw new RequestErrorException(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_IS_ACTIVE'));
        }

        /** @var string $quizOption The submitted option */
        $quizOption = ntGetRequestInput('option');
        /** @var string $correct This is a correct answer or not*/
        $correct = ntGetRequestInput('correct');

        $optionNum = QuizOptionQuery::create()
            ->filterByQuestionID($this->getWrapped()->getQuestionID())
            ->count($con);

        //If too many options have been created.
        if ($optionNum >= ntLoadGlobalConfig('max_option_per_question')) {
            throw new RequestErrorException(HttpCodes::HTTP_NOT_FOUND, 'Too much options added.');
        }

        $question = $this->getWrapped();

        // Create a new option
        $option = new \Nutty\Models\QuizOption();
        $option->setQuestionID($question->getQuestionID());
        $option->setQuizOption($quizOption);

        //Try to find an option with the highest order number.
        $highest = QuizOptionQuery::create()
            ->filterByQuestionID($question->getQuestionID())
            ->orderByQuizOrder(Criteria::DESC)
            ->findOne($con);

        //If not found, new order number is 0
        if (!$highest) {
            $order = 0;
        } else {
            //If found, new order number = highest + 1
            $order = $highest->getQuizOrder() + 1;
        }

        $option->setQuizOrder($order);

        $option->save($con);

        // If this is an right answer or the question doesn't have a right answer,
        // Let the newly created option to be the correct answer.
        if ($question->getRightANS() == -1 || $correct == '1') {
            $question->setRightANS($order);
            $question->save($con);
        }

        return $order;
    }

    /**
     * For deleting an option
     *
     * @param QuizOption $option
     * @param ConnectionInterface|null $con
     *
     * @throws RequestErrorException
     */
    public function deleteOption(QuizOption $option, ConnectionInterface $con = null)
    {
        if ((new StaffHuntWrapper($this->getWrapped()->getHunt()))->isActive()) {
            throw new RequestErrorException(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_IS_ACTIVE'));
        }

        $optionOrder = $option->getQuizOrder();
        $question = $this->getWrapped();

        //If this is a correct answer
        if ($optionOrder == $question->getRightANS()) {
            $question->setRightANS(-1);
            $question->save($con);
        }

        $option->delete($con);
    }

    /**
     * For deleting a question
     *
     * @param ConnectionInterface|null $con
     * @throws RequestErrorException
     */
    public function deleteQuestion(ConnectionInterface $con = null) {
        if ((new StaffHuntWrapper($this->getWrapped()->getHunt()))->isActive()) {
            throw new RequestErrorException(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_IS_ACTIVE'));
        }

        $this->getWrapped()->delete($con);
    }
}