<?php
/**
 * Wrapper class for Hunt Model
 *
 * @author Congyu Fan
 */

namespace App\Models\StaffDefaults;


use App\Misc\Exception\RequestErrorException;
use App\Models\HuntWrapper;
use Kayex\HttpCodes;
use Nutty\Models\Hunt;
use Nutty\Models\QuizQuestionQuery;
use Nutty\Models\Team;
use App\Misc\API\KeyGenerator;
use Nutty\Models\TeamQuery;
use Nutty\Models\GameRecord;
use Nutty\Models\HuntQuery;
use Nutty\Models\Location;
use Nutty\Models\LocationQuery;
use Propel\Runtime\ActiveQuery\Criteria;
use Nutty\Models\QuizQuestion;
use Propel\Runtime\Connection\ConnectionInterface;


/**
 * Class StaffHuntWrapper
 * @package App\Models\StaffDefaults
 */
class StaffHuntWrapper extends HuntWrapper
{
    /**
     * StaffHuntWrapper constructor.
     * @param Hunt $hunt
     */
    public function __construct(Hunt $hunt)
    {
        parent::__construct($hunt);
    }

    /**
     * Create a hunt
     *
     * @param ConnectionInterface|null $con
     * @throws RequestErrorException
     */
    public static function createHunt(ConnectionInterface $con)
    {
        // If get the number of hunts created by the current account.
        $huntNum = HuntQuery::create()
            ->filterByCreatorID(Auth::getIdentification())
            ->count($con);

        //If hunt number exceeds the max.
        if ($huntNum >= ntLoadGlobalConfig('max_hunt_per_staff')) {
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, 'Too many hunts created');
        }

        /** @var string $huntName hunt name*/
        $huntName = ntGetRequestInput('name');
        /** @var string $duration duration*/
        $duration = ntGetRequestInput('duration');
        /** @var int $maxTeam max team number $maxTeam */
        $maxTeam = ntGetRequestInput('max_team');
        /** @var  $difficulty */
        $difficulty = ntGetRequestInput('difficulty');

        //create a new hunt and save
        $hunt = new Hunt();
        $hunt->setHuntName($huntName);
        $hunt->setCreatorID(Auth::getIdentification());
        $hunt->setDuration($duration);
        $hunt->setDifficulty($difficulty);
        $hunt->save($con);

        //Create teams
        $i = 0;
        while ($i < $maxTeam) {
            // Generate a new password
            $pwd = KeyGenerator::genTeamPassword();
            //If the password has already benn used
            if (TeamQuery::create()
                ->filterByTeamPWD($pwd)
                ->exists($con)) {
                continue;
            }

            // Create a new team.
            $team = new Team();
            $team->setTeamPWD($pwd);
            //$team->setTeamName('Team '.($i + 1));
            $team->setHuntID($hunt->getHuntID());
            $team->setLocOffset($i);
            $team->save($con);

            //Create game record for that team.
            $gameRecord = new GameRecord();
            $gameRecord->setTeamID($team->getTeamID());
            $gameRecord->save($con);
            ++$i;
        }
    }


    /**
     * Reset the hunt.
     *
     * @param ConnectionInterface|null $con
     */
    public function reset(ConnectionInterface $con = null)
    {
        $hunt = $this->getWrapped();
        $hunt->setStartTIme(0);     //Reset start time.
        $hunt->save($con);

        //Find teams
        $teams = TeamQuery::create()
            // ->joinWithGameRecord()
            ->filterByHuntID($hunt->getHuntID())
            ->find($con);

        //Reset each team
        foreach ($teams as $team) {
            $teamWrapper = new StaffTeamWrapper($team);

            while (true) {
                //Generate a new password
                $newPWD = KeyGenerator::genTeamPassword();
                // If the new password is not used.
                if (!TeamQuery::create()
                    ->filterByTeamPWD($newPWD)
                    ->exists($con)
                ) {
                    //Reset the team with the generated password
                    $teamWrapper->resetTeam($newPWD, $con);
                    break;
                }
            }
        }
    }

    /**
     * Start a hunt
     *
     * @param ConnectionInterface|null $con
     * @throws RequestErrorException
     */
    public function startHunt(ConnectionInterface $con = null)
    {
        $hunt = $this->getWrapped();

        //If the hunt needs reset
        if ($this->getTimeLeft() != -1) {
            throw new RequestErrorException(HttpCodes::HTTP_CONFLICT, 'The hunt needs to be reset');
        }

        //Make sure there are at least 2 locations associated with this hunt.
        $locations = LocationQuery::create()
            ->filterByHuntID($hunt->getHuntID())
            ->orderByLocationOrder()
            ->count($con);

        //If no enough location.
        if ($locations < 2) {
            throw new RequestErrorException(HttpCodes::HTTP_CONFLICT, 'Not enough locations defined');
        }

        //Check if there's any question without an answer
        /*$noOptQuestion = QuizQuestionQuery::create()
            ->filterByHuntID($hunt->getHuntID())
            ->leftJoinQuizOption()
            ->where('QuizOption.QuestionID IS NULL')
            ->exists($con);*/

        //Find questions with less than 2 options
        $noOptQuestion = QuizQuestionQuery::create()
            ->filterByHuntID($hunt->getHuntID())
            ->innerJoinQuizOption()
            ->groupByQuestionID()
            ->having('COUNT(*) < 2')
            ->exists($con);

        //If exist
        if ($noOptQuestion) {
            throw new RequestErrorException(HttpCodes::HTTP_CONFLICT, 'There exists quiz question without enough option');
        }

        //Get teams
        $teams = TeamQuery::create()
            ->filterByHuntID($hunt->getHuntID())
            ->find($con);

        foreach ($teams as $team) {
            $teamWrapper = new StaffTeamWrapper($team);
            // For each team, we need to calculate the total distance
            $team->setTotalDist($teamWrapper->calTotalDist());
            // Reset the game record
            $teamWrapper->resetGameRecord($con);
            $team->save($con);
        }

        //Change the start time of the hunt to current timestamp.
        $hunt->setStartTIme(time());
        $hunt->save($con);
    }

    /**
     * Add a new location
     *
     * @param ConnectionInterface|null $con
     * @return int The order number of the newly created location
     * @throws RequestErrorException
     */
    public function addNewLoc(ConnectionInterface $con = null)
    {
        // Get the number of created locations for this hunt.
        $locNum = LocationQuery::create()
            ->filterByHuntID($this->getWrapped()->getHuntID())
            ->count($con);

        // If too many created
        if ($locNum >= ntLoadGlobalConfig('max_location_per_hunt')) {
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, 'Too many locations added for the hunt');
        }

        /** @var string $name Location name */
        $name = ntGetRequestInput('name');
        /** @var string $clue Text clue*/
        $clue = ntGetRequestInput('clue');
        /** @var string $latitude */
        $latitude = ntGetRequestInput('latitude');
        /** @var string $longitude */
        $longitude = ntGetRequestInput('longitude');

        $hunt = $this->getWrapped();

        $data = [
            'Name' => $name,
            'Clue' => $clue,
            'Latitude' => $latitude,
            'Longitude' => $longitude,
        ];

        // Create a new location
        $location = new Location();
        // Load data from array.
        $location->fromArray($data);
        // Set hunt id
        $location->setHuntID($hunt->getHuntID());

        // Try to find the location with the highest order number.
        $lastLoc = LocationQuery::create()
            ->filterByHuntID($hunt->getHuntID())
            ->orderByLocationOrder(Criteria::DESC)
            ->findOne($con);

        // If found, order number = highest order number + 1, otherwise 0
        $order = $lastLoc ? $lastLoc->getLocationOrder() + 1 : 0;
        $location->setLocationOrder($order);
        $location->save($con);

        // Return the order id
        return $order;
    }

    /**
     * Add a new question.
     *
     * @param ConnectionInterface|null $con
     * @return int The id of the newly created question
     * @throws RequestErrorException
     */
    public function addNewQuestion(ConnectionInterface $con = null)
    {
        $questionNum = QuizQuestionQuery::create()
            ->filterByHuntID($this->getWrapped()->getHuntID())
            ->count($con);

        // Check if too many questions are created.
        if ($questionNum >= ntLoadGlobalConfig('max_question_per_quiz')) {
            throw new RequestErrorException(HttpCodes::HTTP_BAD_REQUEST, 'Too much questions created for the hunt');
        }

        /** @var string $question question*/
        $question = ntGetRequestInput('quiz_question');
        $quizQuestion = new QuizQuestion();

        $quizQuestion->setHuntID($this->getWrapped()->getHuntID());
        $quizQuestion->setQuestion($question);
        $quizQuestion->save($con);

        // Return the question id of the newly created question
        return $quizQuestion->getQuestionID();
    }

    /**
     * Get the location list of the current hunt.
     *
     * @return array
     */
    public function getLocationInfo()
    {
        $hunt = $this->getWrapped();
        $locations = LocationQuery::create()
            ->filterByHuntID($hunt->getHuntID())
            ->orderByLocationOrder(Criteria::ASC)
            ->find();

        $ret = [];

        foreach ($locations as $location) {
            array_push(
                $ret,
                [
                    'order' => $location->getLocationOrder(),
                    'name' => $location->getName(),
                    'clue' => $location->getClue(),
                    'key' => KeyGenerator::locationKey($location),
                    'latitude' => $location->getLatitude(),
                    'longitude' => $location->getLongitude()
                ]
            );
        }

        return $ret;
    }

    /**
     * Get the question list of the current hunt.
     *
     * @return array
     */
    public function getQuestions()
    {
        $questions = $this->getWrapped()->getQuizQuestions();

        $ret = [];
        foreach ($questions as $question) {
            $optionsList = [];
            $options = $question->getQuizOptions();

            foreach ($options as $option) {
                array_push(
                    $optionsList,
                    [
                        'order' => $option->getQuizOrder(),
                        'correct' => (int)($option->getQuizOrder() == $question->getRightANS()),
                        'option' => $option->getQuizOption()
                    ]);
            }

            array_push(
                $ret,
                [
                    'question_id' => $question->getQuestionID(),
                    'question' => $question->getQuestion(),
                    'option' => $optionsList
                ]);
        }

        return $ret;
    }
}