<?php
/**
 * A wrapper class for the Hunt object. Provide extra functionality.
 *
 * @author Congyu Fan
 * @since Mar 23, 2017
 */

namespace App\Models;

use Nutty\Models\Base\Hunt;

/**
 * Class HuntWrapper
 * @package App\Models
 */
class HuntWrapper extends AbstractWrapper
{

    /**
     * HuntWrapper constructor.
     * @param Hunt $hunt A hunt object.
     */
    public function __construct(Hunt $hunt)
    {
        $this->wrapped = $hunt;
    }

    /**
     * @return Hunt
     */
    public function getWrapped()
    {
        return $this->wrapped;
    }

    /**
     * Get the remaining time of the hunt in seconds.
     *
     * @return int The remaining time in seconds. -1 if not yet started. 0 if finished.
     *
     * @throws \RuntimeException If the start time of the hunt is > than the current timestamp
     *                          but not 0
     */
    public function getTimeLeft()
    {
        /** @var int $startTime The start time of the hunt*/
        $startTime = $this->wrapped->getstartTIme();

        //if not yet started.
        if (!$startTime) {
            return -1;
        }

        /** @var int $now current timestamp*/
        $now = time();
        $duration = $this->wrapped->getduration();

        //If $now < $startTime. there must be sth wrong.
        if ($now < $startTime) {
            throw new \RuntimeException('The start time of the hunt is later than the current time. For a hunt which is
                not yet started, 0 should be the start time.
            ');
        }

        /**
         * if the hunt is finished.
         */
        if (($startTime + $duration) < $now) {
            return 0;
        }

        //calculate the remaining time
        return $duration - ($now - $startTime);
    }

    /**
     * Check if the hunt is active.
     * @return bool
     */
    public function isActive()
    {
        return $this->getTimeLeft() > 0;
    }
}