<?php
/**
 * Utilities for the validation of the input data.
 *
 * @author Congyu Fan
 * @since April 1, 2017
 */
namespace App\Misc\Validation;

use Respect\Validation\Validator as V;

/**
 * Class StaffValidate
 * @package App\Misc\Auth
 */
class Validation
{
    /**
     * To validate if the given is a valid email address.
     *
     * @param string $email An email address
     * @return bool
     */
    public static function email($email)
    {
        return V::email()->validate($email) && V::stringType()->length(1, 254)->validate($email);
    }

    /**
     * To validate if the given is a valid password.
     * The length of a password should be between 8 and 16.
     * A password should consist of alphabets, numbers and ~`!@#$%^&*()_+=-':;./
     *
     * @param string $pwd A password
     * @return bool
     */
    public static function staffPassword($pwd)
    {
        return V::stringType()->length(8, 16)->Alnum('~`!@#$%^&*()_+=-\':;./')->validate($pwd);
    }

    /**
     * Validate the coordinates.
     *
     * @param mixed $latitude
     * @param mixed $longitude
     * @return bool
     */
    public static function coordinates($latitude, $longitude)
    {
        $floatValidator = V::floatVal();

        return $floatValidator->between(-90.0, 90.0, true)->validate($latitude) &&
            $floatValidator->between(-180.0, 180.0, true)->validate($longitude);
    }

    /**
     * Validate clue text for a location.
     *
     * @param $clue
     * @return bool
     */
    public static function clueText($clue)
    {
        return V::stringType()->length(1, 120)->validate($clue);
    }

    /**
     * Validate the name of a location.
     *
     * @param $name
     * @return bool
     */
    public static function locName($name)
    {
        $val = V::stringType()->length(1, 20)->validate($name);
        return $val;
    }

    /**
     * Validate hunt information
     *
     * @param $name
     * @param $duration
     * @param $maxTeam
     * @param $difficulty
     * @return bool
     */
    public static function huntInfo($name, $duration, $maxTeam, $difficulty)
    {
        return V::stringType()->length(1, 20)->validate($name)
                && V::intVal()->positive()->validate($duration)
                && V::intVal()->between(1, ntLoadGlobalConfig('max_team_per_hunt'), true)->validate($maxTeam)
                && V::intVal()->between(0, 1, true)->validate($difficulty)
            ;
    }

    /**
     * Validate a quiz question
     *
     * @param $question
     * @return bool
     */
    public static function quizQuestion($question)
    {
        return V::stringType()->length(1, 100)->validate($question);
    }

    /**
     * Validate a quiz option
     *
     * @param $option
     * @return bool
     */
    public static function quizOption($option) {
        return V::stringType()->length(1, 20)->validate($option);
    }

    /**
     * Validate team name
     *
     * @param $teamName
     * @return bool
     */
    public static function teamName($teamName)
    {
        return V::stringType()->length(1, 15)->validate($teamName);
    }

    /**
     * Check bad words.
     *
     * @param $text
     * @return bool true if bad words found.
     */
    public static function badWords($text)
    {
        return App::make('badWord')->setText($text)->checkAmong();
    }
}