<?php
/**
 * Validation for staff quiz option APIs
 *
 * @author Congyu Fan
 */

namespace App\Misc\Validation;

use Respect\Validation\Validator as V;


/**
 * Class StaffQuizOptionValidate
 * @package App\Misc\Validation
 */
class StaffQuizOptionValidate implements InputValidateInterface
{
    /**
     * Validate question id
     * @param $errMsg
     * @return bool
     */
    private function validateQuestionId(&$errMsg)
    {
        if (!V::intVal()->Min(0)->validate(ntGetRequestInput('question_id'))) {
            $errMsg = 'Invalid question id';
            return false;
        }

        return true;
    }

    /**
     * Validate order number
     * @param $errMsg
     * @return bool
     */
    private function validateOrder(&$errMsg)
    {
        if (!V::intVal()->Min(0)->validate(ntGetRequestInput('order'))) {
            $errMsg = 'Invalid order';
            return false;
        }

        return true;
    }

    /**
     * Validate an option
     * @param $errMsg
     * @return bool
     */
    private function validateOption(&$errMsg)
    {
        $opt = ntGetRequestInput('option');
        if (!Validation::quizOption($opt)) {
            $errMsg = 'Invalid option';
            return false;
        }

        if (Validation::badWords($opt)) {
            $errMsg = 'Option'.ntGetStringLiteral('CONTAIN_BAD_WORD');
            return false;
        }

        return true;
    }

    /**
     * For option creating.
     * @param $errMsg
     * @return bool
     */
    private function validateCreate(&$errMsg)
    {
        if (!$this->validateQuestionId($errMsg)) {
            return false;
        }

        if (!$this->validateOption($errMsg)) {
            return false;
        }

        return true;
    }

    /**
     * For option deleting
     *
     * @param $errMsg
     * @return bool
     */
    private function validateDelete(&$errMsg)
    {
        if (!$this->validateQuestionId($errMsg)) {
            return false;
        }

        if (!$this->validateOrder($errMsg)) {
            return false;
        }

        return true;
    }

    /**
     * For setting a correct answer.
     *
     * @param $errMsg
     * @return bool
     */
    private function validateSetCorrect(&$errMsg)
    {
        if (!$this->validateQuestionId($errMsg)) {
            return false;
        }

        if (!$this->validateOrder($errMsg)) {
            return false;
        }

        return true;
    }

    /**
     * For updating an option
     *
     * @param $errMsg
     * @return bool
     */
    private function validateUpdate(&$errMsg)
    {
        if (!$this->validateQuestionId($errMsg)) {
            return false;
        }

        if (!Validation::quizQuestion(ntGetRequestInput('quiz_question'))) {
            $errMsg = 'Invalid question';
            return false;
        }

        return true;
    }

    public function validate($action, &$errMsg)
    {
        switch ($action) {
            case 'create':  //Creating an option
                return $this->validateCreate($errMsg);
            case 'delete':  //Deleting an option
                return $this->validateDelete($errMsg);
            case 'set_correct': //Setting a correct answer.
                return $this->validateSetCorrect($errMsg);
            case 'update':  //update an option
                return $this->validateUpdate($errMsg);
            default:
                $errMsg = ntGetStringLiteral('INVALID_ACTION');
                return false;
        }
    }
}