<?php
/**
 * Validation for staff location management APIs
 *
 * @author Congyu Fan
 */

namespace App\Misc\Validation;


use Respect\Validation\Validator as V;

/**
 * Class StaffLocationManageValidate
 * @package App\Misc\Validation
 */
class StaffLocationManageValidate implements InputValidateInterface
{
    /**
     * Validate hunt id
     * @param string $errMsg
     * @return bool
     */
    private function validateHuntID(&$errMsg)
    {
        if (!V::intVal()->Min(0)->validate(ntGetRequestInput('hunt_id'))) {
            $errMsg = 'Invalid hunt id';
            return false;
        }
        return true;
    }

    /**
     * For updating location information.
     *
     * @param $errMsg
     * @return bool|string
     */
    private function validateUpdate(&$errMsg)
    {
        // Validate hunt id
        if (!$this->validateHuntID($errMsg)) {
            return false;
        }

        // Validate order number
        if (!V::intVal()->Min(0)->validate(ntGetRequestInput('order'))) {
            $errMsg = 'Invalid order number';
            return false;
        }

        $name = ntGetRequestInput('name');
        if ($name) {    //If submitted a new name
            if (!Validation::locName($name)) {
                $errMsg = 'Invalid location name';
                return false;
            }

            if (Validation::badWords($name)) {
                $errMsg = 'Name '.ntGetStringLiteral('CONTAIN_BAD_WORD');
                return false;
            }
        }

        $clue = ntGetRequestInput('clue');

        if ($clue) {    //If submitted a new clue.
            if (!Validation::clueText($clue)) {
                return 'Invalid clue';
            }

            if (Validation::badWords($clue)) {
                $errMsg = 'Clue '.ntGetStringLiteral('CONTAIN_BAD_WORD');
                return false;
            }
        }

        $latitude = ntGetRequestInput('latitude');
        $longitude = ntGetRequestInput('longitude');
        if ($latitude && $longitude) {  //If submitted new coordinates.
            if (!Validation::coordinates($latitude, $longitude)) {
                return 'Invalid coordinates';
            }
        }

        return true;
    }

    /**
     * For hunt creation.
     * @param $errMsg
     * @return bool
     */
    private function validateCreate(&$errMsg)
    {
        //Validate hunt id
        if (!$this->validateHuntID($errMsg)) {
            return false;
        }

        $name = ntGetRequestInput('name');
        $clue = ntGetRequestInput('clue');

        //Validate name
        if (!Validation::locName($name)) {
            $errMsg = 'Invalid location name';
            return false;
        }

        if (Validation::badWords($name)) {
            $errMsg = 'Name '.ntGetStringLiteral('CONTAIN_BAD_WORD');
            return false;
        }

        //Validate clue
        if (!Validation::clueText($clue)) {
            $errMsg = 'Invalid clue';
            return false;
        }

        if (Validation::badWords($clue)) {
            $errMsg = 'Clue '.ntGetStringLiteral('CONTAIN_BAD_WORD');
            return false;
        }

        //Validate coordinates
        if (!Validation::coordinates(ntGetRequestInput('latitude'), ntGetRequestInput('longitude'))) {
            $errMsg =  'Invalid coordinates';
            return false;
        }

        return true;
    }

    /**
     * For remove operation.
     *
     * @param $errMsg
     * @return bool
     */
    private function validateRemove(&$errMsg)
    {
        if (!$this->validateHuntID($errMsg)) {
            return false;
        }

        if (!V::intVal()->Min(0)->validate(ntGetRequestInput('order'))) {
            $errMsg = 'Invalid order';
            return false;
        }
        return true;
    }

    /**
     * @inheritdoc
     */
    public function validate($action, &$errMsg)
    {
        switch ($action) {
            case 'retrieve':    //Retrieve location list
                return $this->validateHuntID($errMsg);
            case 'update':  //Update a location
                return $this->validateUpdate($errMsg);
            case 'create':  //Create a location
                return $this->validateCreate($errMsg);
            case 'delete':  //delete a location
                return $this->validateRemove($errMsg);
            default: {
                $errMsg = ntGetStringLiteral('INVALID_ACTION');
                return false;
            }
        }
    }
}