<?php
/**
 * Validation for location template management APIs
 *
 * @author Congyu Fan
 */

namespace App\Misc\Validation;


use Respect\Validation\Validator as V;

class StaffLocTemplateValidate implements InputValidateInterface
{
    /**
     * Validate template id
     *
     * @return bool
     */
    private function validateTemplateId()
    {
        return V::intVal()->Min(0)->validate(ntGetRequestInput('template_id'));
    }

    /**
     * Validate hunt create
     *
     * @param $errMsg
     * @return bool
     */
    private function validateCreate(&$errMsg)
    {
        $name = ntGetRequestInput('name');
        $clue = ntGetRequestInput('clue');
        $latitude = ntGetRequestInput('latitude');
        $longitude = ntGetRequestInput('longitude');

        //Validate location name
        if (!Validation::locName($name)) {
            $errMsg = 'Illegal Location Name';
            return false;
        }

        //Validate clue text
        if (!Validation::clueText($clue)) {
            $errMsg = 'Illegal Text Clue';
            return false;
        }

        //Check bad words
        if (Validation::badWords($name)) {
            $errMsg = 'Location name'.ntGetStringLiteral('CONTAIN_BAD_WORD');
            return false;
        }

        if (Validation::badWords($clue)) {
            $errMsg = 'Clue '.ntGetStringLiteral('CONTAIN_BAD_WORD');
            return false;
        }

        //validate coordinates
        if (!Validation::coordinates($latitude, $longitude)) {
            $errMsg = 'Invalid coordinates';
            return false;
        }

        return true;
    }

    /**
     * For updating.
     *
     * @param $errMsg
     * @return bool
     */
    private function validateUpdate(&$errMsg)
    {
        //validate template id
        if (!$this->validateTemplateId())  {
            $errMsg = 'Invalid template id';
            return false;
        }

        $name = ntGetRequestInput('name');
        //If there exists new name
        if ($name) {
            if (!Validation::locName($name)) {
                $errMsg = 'Invalid location name';
                return false;
            }

            if (Validation::badWords($name)) {
                $errMsg = 'Location name '.ntGetStringLiteral('CONTAIN_BAD_WORD');
                return false;
            }
        }

        $clue = ntGetRequestInput('clue');
        //If new clue submitted.
        if ($clue) {
            if (!Validation::clueText($clue)) {
                $errMsg = 'Invalid Text Clue';
                return false;
            }

            if (Validation::badWords($clue)) {
                $errMsg = 'Clue'.ntGetStringLiteral('CONTAIN_BAD_WORD');
                return false;
            }
        }

        $latitude = ntGetRequestInput('latitude');
        $longitude = ntGetRequestInput('longitude');
        // If coordinates submitted.
        if ($latitude && $longitude) {
            if (!Validation::coordinates($latitude, $longitude)) {
                $errMsg = 'Invalid Coordinates';
                return false;
            }
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function validate($action, &$errMsg)
    {
        switch ($action) {
            case 'create':  //For creating
                return $this->validateCreate($errMsg);
            case 'update':  //For updating
                return $this->validateUpdate($errMsg);
            case 'delete': { //For deleting
                if (!$this->validateTemplateId())  {
                    $errMsg = 'Invalid template id';
                    return false;
                }

                $errMsg = null;
                return true;
            }
            case 'retrieve': {  //For retrieving template list.
                return true;
            }
            default: {
                $errMsg = ntGetStringLiteral('INVALID_ACTION');
                return false;
            }
        }
    }
}