<?php
/**
 * Staff hunt management APIs input validation
 *
 * @author Congyu Fan
 */
namespace App\Misc\Validation;


use Respect\Validation\Validator as V;

/**
 * Class StaffHuntManageValidate
 * @package App\Misc\Validation
 */
class StaffHuntManageValidate implements InputValidateInterface
{
    /**
     * Validate hunt id
     *
     * @param $errMsg
     * @return bool
     */
    private function validateHuntId(&$errMsg)
    {
        $huntId = ntGetRequestInput('hunt_id');
        if ((!$huntId) || !V::intVal()->Min(0)->validate($huntId)) {
            $errMsg = 'Invalid hunt id';
            return false;
        }

        return true;
    }

    /**
     * For hunt create requests
     *
     * @param $errMsg
     * @return bool
     */
    private function validateCreate(&$errMsg)
    {
        $name = ntGetRequestInput('name');

        //Validate hunt name.
        if (!V::stringType()->length(1, 20)->validate($name)) {
            $errMsg = 'Invalid hunt name';
            return false;
        }

        //Check bad words
        if (Validation::badWords($name)) {
            $errMsg = 'Name '.ntGetStringLiteral('CONTAIN_BAD_WORD');
            return false;
        }

        // Duration must be a positive int.
        if (!V::intVal()->positive()->validate(ntGetRequestInput('duration'))) {
            $errMsg = 'Invalid hunt duration';
            return false;
        }

        // max team number
        if (!V::intVal()->between(1, ntLoadGlobalConfig('max_team_per_hunt'), true)->validate(ntGetRequestInput('max_team'))) {
            $errMsg = 'Invalid max team number';
            return false;
        }

        // Difficulty must be either 0 or 1
        if (!V::intVal()->between(0, 1, true)->validate(ntGetRequestInput('difficulty'))) {
            $errMsg = 'Invalid difficulty';
            return false;
        }

        return true;
    }


    /**
     * For distributing password by email.
     *
     * @param $errMsg
     * @return bool
     */
    private function validateDistByEmail(&$errMsg)
    {
        //validate team id
        if (!V::intVal()->Min(0)->validate(ntGetRequestInput('team_id'))) {
            $errMsg = 'Invalid team id';
            return false;
        }

        //validate email address
        if (!Validation::email(ntGetRequestInput('email_addr'))) {
            $errMsg = 'Invalid email';
            return false;
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function validate($action, &$errMsg)
    {
        switch ($action) {
            case 'create': //For creating hunt
                return $this->validateCreate($errMsg);
            case 'reset':   //For resetting a hunt
                return $this->validateHuntId($errMsg);
            case 'start':   //for starting a hunt
                return $this->validateHuntId($errMsg);
            case 'retrieve':    //for retrieving hunt list
                return true;
            case 'retrieve_one':    //For retrieving a specific hunt.
                return $this->validateHuntId($errMsg);
            case 'delete':  //For deleting a hunt.
                return $this->validateHuntId($errMsg);
            case 'retrieve_teams': //For retrieving team list of a specific hunt.
                return $this->validateHuntId($errMsg);
            case 'dist_email':  //For distributing passwords by email.
                return $this->validateDistByEmail($errMsg);
            default:
                $errMsg = ntGetStringLiteral('INVALID_ACTION');
                return false;
        }
    }
}