<?php
/**
 * The staff login module.
 *
 * @author Congyu Fan
 * @since Apr 1, 2017
 */

namespace App\Misc\Auth;

use App\Misc\API\KeyGenerator;
use App\Misc\Exception\RequestErrorException;
use CodeMommy\CookiePHP\Cookie;
use Kayex\HttpCodes;
use Nutty\Models\StaffQuery;
use Propel\Runtime\Connection\ConnectionInterface;

/**
 * Class StaffLogin
 * @package App\Misc\Auth
 */
class StaffLogin
{
    /**
     * Attempt to login.
     *
     * @param ConnectionInterface $con
     * @return void
     * @throws RequestErrorException If failed to login
     */
    public static function login(ConnectionInterface $con = null)
    {
        // delete cookie data.
        if (Cookie::isExist('email')) {
            Cookie::delete('email');
        }

        if (Cookie::isExist('ticket')) {
            Cookie::delete('ticket');
        }

        $email = ntGetRequestInput('email');
        $password = ntGetRequestInput('password');

        // Find staff by email
        $staff = StaffQuery::create()
            ->findOneByEmail($email);

        // If not found.
        if (!$staff) {
            throw new RequestErrorException(HttpCodes::HTTP_UNAUTHORIZED, ntGetStringLiteral('INVALID_LOGIN_INFORMATION'));
        }

        // Generate the password with the subtmitted information.
        $userPass = KeyGenerator::genPassword($password, $staff->getSalt());

        // Compare with the correct password.
        if ($userPass !== $staff->getPassword()) {
            throw new RequestErrorException(HttpCodes::HTTP_UNAUTHORIZED, ntGetStringLiteral('INVALID_LOGIN_INFORMATION'));
        }

        // Generate a new token and store into the DB
        $newToken = KeyGenerator::genLoginToken();
        $staff->setToken($newToken)->save($con);

        // Generate a login ticket
        $ticket = KeyGenerator::genLoginTicket($staff);

        // Store cookies.
        $expire = ntLoadGlobalConfig('staff_cookie_expire');
        Cookie::set('email', $staff->getEmail(), time() + $expire, ntGetRootDir());
        Cookie::set('ticket', $ticket, time() + $expire, ntGetRootDir());
        $_COOKIE['email'] = $staff->getEmail();
        $_COOKIE['ticket'] = $ticket;
    }

    /**
     * Attempt to logout
     */
    public static function logout()
    {
        // delete cookies
        if (Cookie::isExist('email')) {
            Cookie::delete('email');
        }

        if (Cookie::isExist('ticket')) {
            Cookie::delete('ticket');
        }

        Cookie::set('email', NULL, time() - 3600, ntGetRootDir());
        Cookie::set('ticket', NULL, time() - 3600, ntGetRootDir());
        unset($_COOKIE['email']);
        unset($_COOKIE['ticket']);
    }
}