<?php
/**
 * The authentication module for the staff.
 *
 * @author Congyu Fan
 * @since Apr 1, 2017
 */
namespace App\Misc\Auth;

use App\Misc\API\KeyGenerator;
use App\Misc\Validation\Validation;
use Bistro\Session\Session;
use CodeMommy\CookiePHP\Cookie;
use Nutty\Models\StaffQuery;

class StaffAuth implements AuthInterface
{
    /** @var bool $authenticated If the user is authenticated. */
    private $authenticated = false;
    /** @var int $id staff id */
    private $id = 0;
    /** @var string email address of the staff. */
    private $email = '';

    /**
     * StaffAuth constructor.
     */
    public function __construct()
    {
    }

    /**
     * @inheritDoc
     */
    public function isAuthenticated()
    {
        //If already authenticated.
        if ($this->authenticated) {
            return true;
        }

        /**
         * Check cookies. If no login information, return false.
         */
        if (!(Cookie::isExist('email') && Cookie::isExist('ticket'))) {
            return false;
        }

        $email = Cookie::get('email');
        // validate email address
        if (!Validation::email($email)) {
            return false;
        }

        // Find the staff by email.
        $staff = StaffQuery::create()
            ->findOneByEmail($email);

        // If not found
        if (!$staff) {
            return false;
        }

        // Generate the correct login ticket for the current user.
        $realTicket = KeyGenerator::genLoginTicket($staff);

        // Compare with the submitted ticket.
        if ($realTicket !== Cookie::get('ticket')) {
            return false;
        }

        // Store staff information.
        $this->id = $staff->getStaffID();
        $this->email = $staff->getEmail();
        $this->authenticated = true;

        // Renew cookie data.
        $expire = ntLoadGlobalConfig('staff_cookie_expire');
        Cookie::set('email', $this->email, time() + $expire, ntGetRootDir());
        Cookie::set('ticket', $realTicket, time() + $expire, ntGetRootDir());

        return true;
    }

    /**
     * @inheritdoc
     *
     * @return string|null The email address of the staff. null if not authenticated.
     */
    public function getName()
    {
        if (!$this->isAuthenticated()) {
            return null;
        }

        return $this->email;
    }

    /**
     * @inheritdoc
     *
     * @return int|null Staff id. null if not authenticated.
     */
    public function getIdentification()
    {
        if (!$this->isAuthenticated()) {
            return null;
        }

        return $this->id;
    }
}