<?php
/**
 * Controller for location related team APIs
 *
 * @author Congyu Fan
 * @since Mar 30, 2017
 */

namespace App\Controllers\API;


use App\Misc\Exception\RequestErrorException;
use Kayex\HttpCodes;
use Nutty\Controller\AbstractController;
use App\Models\TeamDefaults\TeamTeamWrapper;
use App\Models\TeamDefaults\TeamHuntWrapper;
use App\Misc\API\APIResponseGenerator;
use LucDeBrouwer\Distance\Distance;
use Nutty\Models\Map\GameRecordTableMap;
use Propel\Runtime\Connection\ConnectionInterface;
use Propel\Runtime\Exception\PropelException;
use Propel\Runtime\Propel;


class TeamAPILocationController extends AbstractController
{
    /**
     * Handles the queries for the current location of the current team.
     *
     * @param ConnectionInterface $con
     * @param TeamTeamWrapper $teamWrapper The current team.
     * @param TeamHuntWrapper $huntWrapper The current hunt.
     *
     * @return string
     */
    public function retrieve(ConnectionInterface $con, TeamTeamWrapper $teamWrapper, TeamHuntWrapper $huntWrapper)
    {
        //if the hunt is inactive, return an error message.
        if (!$huntWrapper->isActive()) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_INACTIVE'));
        }

        $ret = $teamWrapper->retrieveLoc($huntWrapper, $con);

        return APIResponseGenerator::genArrayJson($ret);
    }

    /**
     * Will be invoked when a team reaches a location
     *
     * @param ConnectionInterface $con
     * @param TeamTeamWrapper $teamWrapper  Wrapper class for the current Team
     * @param TeamHuntWrapper $huntWrapper  Wrapper class for the current Hunt
     * @param Distance $distanceCal     Distance class to calculate the distance between 2 coordinates.
     *
     * @return string
     */
    public function update(
        ConnectionInterface $con,
        TeamTeamWrapper $teamWrapper,
        TeamHuntWrapper $huntWrapper,
        Distance $distanceCal
    ) {
        //if the hunt is inactive.
        if (!$huntWrapper->isActive()) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_INACTIVE'));
        }

        $ret = $teamWrapper->moveToNextLoc($distanceCal, $con);

        return APIResponseGenerator::genArrayJson($ret);
    }
}