<?php
/**
 * Controller for staff quiz management APIs
 *
 * @author Congyu Fan
 * @since Apr 6, 2017
 */

namespace App\Controllers\API;


use App\Misc\API\APIResponseGenerator;
use App\Models\StaffDefaults\StaffHuntWrapper;
use App\Models\StaffDefaults\StaffQuizQuestionWrapper;
use Kayex\HttpCodes;
use Nutty\Controller\AbstractController;
use Propel\Runtime\Connection\ConnectionInterface;


/**
 * Class StaffQuizQuestionController
 * @package App\Controllers\API
 */
class StaffQuizQuestionController extends AbstractController
{
    /**
     * Create a new question
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper $huntWrapper
     * @return string
     */
    public function create(ConnectionInterface $con, StaffHuntWrapper $huntWrapper)
    {
        // If the hunt is active.
        if ($huntWrapper->isActive()) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_IS_ACTIVE'));
        }

        $newId = $huntWrapper->addNewQuestion($con);

        return APIResponseGenerator::genArrayJson(['question_id' => $newId]);
    }

    /**
     * Update a question.
     *
     * @param ConnectionInterface $con
     * @param StaffQuizQuestionWrapper|null $questionWrapper
     * @return string
     */
    public function update(ConnectionInterface $con, StaffQuizQuestionWrapper $questionWrapper = null)
    {
        if (!$questionWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('QUES_NOT_FOUND'));
        }

        if ((new StaffHuntWrapper($questionWrapper->getWrapped()->getHunt()))->isActive()) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_IS_ACTIVE'));
        }

        $question = ntGetRequestInput('quiz_question');
        $questionWrapper->getWrapped()->setQuestion($question)->save($con);

        return APIResponseGenerator::genArrayJson([]);
    }


    /**
     * Remove a question
     *
     * @param ConnectionInterface $con
     * @param StaffQuizQuestionWrapper|null $questionWrapper
     * @return string
     */
    public function delete(ConnectionInterface $con, StaffQuizQuestionWrapper $questionWrapper = null)
    {
        if (!$questionWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('QUES_NOT_FOUND'));
        }

        $questionWrapper->deleteQuestion($con);

        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Retrieve the question list of a specific hunt.
     *
     * @param StaffHuntWrapper $huntWrapper
     * @return string
     */
    public function retrieve(StaffHuntWrapper $huntWrapper)
    {
        $ret = $huntWrapper->getQuestions();

        return APIResponseGenerator::genArrayJson($ret);
    }

}