<?php
/**
 * The controller for staff quiz question option management APIs
 *
 * @author Congyu Fan
 * @since Apr 7, 2017
 */

namespace App\Controllers\API;


use App\Misc\API\APIResponseGenerator;
use App\Models\StaffDefaults\StaffQuizOptionWrapper;
use App\Models\StaffDefaults\StaffQuizQuestionWrapper;
use Kayex\HttpCodes;
use Nutty\Controller\AbstractController;
use Propel\Runtime\Connection\ConnectionInterface;


/**
 * Class StaffQuizOptionController
 * @package App\Controllers\API
 */
class StaffQuizOptionController extends AbstractController
{
    /**
     * Create a new option
     *
     * @param ConnectionInterface $con
     * @param StaffQuizQuestionWrapper|null $questionWrapper
     * @return string
     */
    public function create(ConnectionInterface $con, StaffQuizQuestionWrapper $questionWrapper = null)
    {
        // If question not found
        if (!$questionWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('QUES_NOT_FOUND'));
        }

        $order = $questionWrapper->addOption($con);

        return APIResponseGenerator::genArrayJson(['order' => $order]);
    }

    /**
     * Delete an option
     *
     * @param ConnectionInterface $con
     * @param StaffQuizQuestionWrapper|null $questionWrapper
     * @param StaffQuizOptionWrapper|null $optionWrapper
     * @return string
     */
    public function delete(ConnectionInterface $con, StaffQuizQuestionWrapper $questionWrapper = null, StaffQuizOptionWrapper $optionWrapper = null)
    {
        // If not found
        if (!$questionWrapper || !$optionWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('OPT_NOT_FOUND'));
        }

        $option = $optionWrapper->getWrapped();
        $question = $questionWrapper->getWrapped();

        // You can't delete a correct option.
        if ($question->getRightANS() == $option->getQuizOrder()) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('OPT_CORRECT'));
        }

        $questionWrapper->deleteOption($option, $con);

        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Set the correct option.
     *
     * @param ConnectionInterface $con
     * @param StaffQuizQuestionWrapper|null $questionWrapper
     * @param StaffQuizOptionWrapper|null $optionWrapper
     * @return string
     */
    public function setCorrect(ConnectionInterface $con, StaffQuizQuestionWrapper $questionWrapper = null, StaffQuizOptionWrapper $optionWrapper = null)
    {
        if (!$questionWrapper || !$optionWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('OPT_NOT_FOUND'));
        }

        $option = $optionWrapper->getWrapped();
        $question = $questionWrapper->getWrapped();

        $question->setRightANS($option->getQuizOrder())->save($con);

        return APIResponseGenerator::genArrayJson([]);
    }
}