<?php
/**
 * Controller for staff location manage APIs.
 *
 * @author Congyu Fan
 * @since Apr 5, 2017
 */

namespace App\Controllers\API;


use App\Misc\API\APIResponseGenerator;
use App\Models\StaffDefaults\StaffHuntWrapper;
use App\Models\StaffDefaults\StaffLocationWrapper;
use Kayex\HttpCodes;
use Nutty\Controller\AbstractController;
use Propel\Runtime\Connection\ConnectionInterface;

/**
 * Class StaffLocManageController
 * @package App\Controllers\API
 */
class StaffLocManageController extends AbstractController
{
    /**
     * Get the location list of a specific hunt
     *
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function queryLocForHunt(StaffHuntWrapper $huntWrapper = null)
    {
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        return APIResponseGenerator::genArrayJson($huntWrapper->getLocationInfo());
    }

    /**
     * Update the location information of a hunt
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper|null $huntWrapper
     * @param StaffLocationWrapper|null $locationWrapper
     * @return string
     */
    public function updateLocation(ConnectionInterface $con, StaffHuntWrapper $huntWrapper = null, StaffLocationWrapper $locationWrapper = null)
    {
        // If location not found.
        if (!$locationWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('LOC_NOT_FOUND'));
        }

        // If the hunt needs reset,
        if ($huntWrapper->getTimeLeft() != -1) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_NEED_RESET'));
        }

        $locationWrapper->updateLoc($con);

        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Create a new location for a specific hunt
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function createLoc(ConnectionInterface $con, StaffHuntWrapper $huntWrapper = null)
    {
        // If hunt not found
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        // Hunt needs reset
        if ($huntWrapper->getTimeLeft() != -1) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_NEED_RESET'));
        }

        $order = $huntWrapper->addNewLoc($con);
        return APIResponseGenerator::genArrayJson(['order' => $order]);
    }

    /**
     * Remove a specific location
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper|null $huntWrapper
     * @param StaffLocationWrapper|null $locationWrapper
     * @return string
     */
    public function removeLoc(ConnectionInterface $con, StaffHuntWrapper $huntWrapper = null, StaffLocationWrapper $locationWrapper = null)
    {
        // If location not found
        if (!$locationWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('LOC_NOT_FOUND'));
        }

        // If hunt needs reset
        if ($huntWrapper->getTimeLeft() != -1) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_NEED_RESET'));
        }

        //delete
        $locationWrapper->getWrapped()->delete($con);
        return APIResponseGenerator::genArrayJson([]);
    }
}