<?php
/**
 * Controller for staff hunt management APIs
 *
 * @author Congyu Fan
 */

namespace App\Controllers\API;


use App\Misc\API\APIResponseGenerator;
use App\Models\StaffDefaults\StaffHuntWrapper;
use App\Models\StaffDefaults\StaffTeamWrapper;
use Kayex\HttpCodes;
use Nutty\Controller\AbstractController;
use Nutty\Models\Base\HuntQuery;
use Propel\Runtime\Connection\ConnectionInterface;


class StaffHuntManageController extends AbstractController
{
    /**
     * Create a hunt
     *
     * @param ConnectionInterface $con
     * @return string
     */
    public function createHunt(ConnectionInterface $con)
    {
        StaffHuntWrapper::createHunt($con);
        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Reset a hunt
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function resetHunt(ConnectionInterface $con, StaffHuntWrapper $huntWrapper = null)
    {
        // If no hunt was found.
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        $huntWrapper->reset($con);

        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Start a hunt.
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function startHunt(ConnectionInterface $con, StaffHuntWrapper $huntWrapper = null)
    {
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        $huntWrapper->startHunt($con);

        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Get the information of hunts created by the current staff
     *
     * @return string
     */
    public function getHunts()
    {
        // Find hunts
        $hunts = HuntQuery::create()
            ->findByCreatorID(Auth::getIdentification());

        /** @var array $ret To store results*/
        $ret = [];

        foreach ($hunts as $hunt) {
            $wrapper = new StaffHuntWrapper($hunt);
            array_push(
                $ret,
                [
                    'hunt_id' => $hunt->getHuntID(),
                    'hunt_name' => $hunt->getHuntName(),
                    'difficulty' => $hunt->getDifficulty(),
                    'time_left' => $wrapper->getTimeLeft()
                ]
            );
        }

        return APIResponseGenerator::genArrayJson($ret);
    }

    /**
     * Remove a hunt
     *
     * @param ConnectionInterface $con
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function removeHunt(ConnectionInterface $con, StaffHuntWrapper $huntWrapper = null)
    {
        // If hunt not found
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        // If the hunt is active, return an error
        if ($huntWrapper->isActive()) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_CONFLICT, ntGetStringLiteral('HUNT_IS_ACTIVE'));
        }

        // Delete it.
        $huntWrapper->getWrapped()->delete($con);
        return APIResponseGenerator::genArrayJson([]);
    }

    /**
     * Get the information of a specific hunt.
     *
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function getHuntInfo(StaffHuntWrapper $huntWrapper = null)
    {
        // If hunt not found
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        $ret = [
            'hunt_name' => $huntWrapper->getWrapped()->getHuntName(),
            'difficulty' => $huntWrapper->getWrapped()->getDifficulty(),
            'time_left' => $huntWrapper->getTimeLeft()
        ];

        return APIResponseGenerator::genArrayJson($ret);
    }

    /**
     * Get team information of a specific hunt.
     *
     * @param StaffHuntWrapper|null $huntWrapper
     * @return string
     */
    public function getHuntTeams(StaffHuntWrapper $huntWrapper = null)
    {
        //If hunt not found
        if (!$huntWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('HUNT_NOT_FOUND'));
        }

        /** @var array $ret To store results*/
        $ret = [];

        // Get the teams of this hunt
        $teams = $huntWrapper->getWrapped()->getTeams();

        foreach ($teams as $team) {
            $teamName = $team->getTeamName();
            array_push($ret,
                [
                    'team_id' => $team->getTeamID(),
                    'team_name' => $teamName ? $teamName : 'unnamed',
                    'team_pwd' => $team->getTeamPWD(),
                    'email' =>$team->getEmail(),
                ]
            );
        }

        return APIResponseGenerator::genArrayJson($ret);
    }

    /**
     * Distribute password of a team by email
     *
     * @param ConnectionInterface $con
     * @param StaffTeamWrapper|null $teamWrapper
     * @return string
     */
    public function distPWDByEmail(ConnectionInterface $con, StaffTeamWrapper $teamWrapper = null)
    {
        // If hunt not found
        if (!$teamWrapper) {
            return APIResponseGenerator::genJsonErrorMsg(HttpCodes::HTTP_NOT_FOUND, ntGetStringLiteral('TEAM_NOT_FOUND'));
        }

        $teamWrapper->distPwdByEmail($con);

        return APIResponseGenerator::genArrayJson([]);
    }
}